<?php
session_start();
include 'layouts/dbconn.php';

// Set timezone to East Africa Time
date_default_timezone_set('Africa/Nairobi');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Fetch current user details
$current_user_id = $_SESSION['user_id'];
$sql_current_user = "SELECT u.*, ur.role_name FROM users u 
                     INNER JOIN user_roles ur ON u.role_id = ur.role_id 
                     WHERE u.user_id = ?";
$stmt_current = $conn->prepare($sql_current_user);
$stmt_current->bind_param("i", $current_user_id);
$stmt_current->execute();
$current_user = $stmt_current->get_result()->fetch_assoc();

// Define system modules and their permissions
$system_modules = [
    'Patient Management' => [
        'view_patients' => 'View Patient Records',
        'add_patients' => 'Register New Patients', 
        'edit_patients' => 'Edit Patient Information',
        'delete_patients' => 'Delete Patient Records',
        'export_patients' => 'Export Patient Data'
    ],
    'Patient Visits' => [
        'view_visits' => 'View Patient Visits',
        'register_visits' => 'Register New Visits',
        'edit_visits' => 'Edit Visit Information',
        'cancel_visits' => 'Cancel Visits',
        'visit_reports' => 'Generate Visit Reports'
    ],
    'Consultations' => [
        'view_consultations' => 'View All Consultations',
        'conduct_consultations' => 'Conduct Consultations',
        'edit_consultations' => 'Edit Consultation Records',
        'consultation_reports' => 'Generate Consultation Reports',
        'view_medical_history' => 'View Patient Medical History'
    ],
    'Laboratory' => [
        'view_lab_orders' => 'View Lab Orders',
        'create_lab_orders' => 'Create Lab Orders',
        'process_lab_tests' => 'Process Lab Tests',
        'enter_lab_results' => 'Enter Lab Results',
        'approve_lab_results' => 'Approve Lab Results',
        'lab_reports' => 'Generate Lab Reports'
    ],
    'Procedures' => [
        'view_procedures' => 'View All Procedures',
        'perform_procedures' => 'Perform Procedures',
        'schedule_procedures' => 'Schedule Procedures',
        'procedure_reports' => 'Generate Procedure Reports'
    ],
    'Billing & Payments' => [
        'view_invoices' => 'View Invoices',
        'create_invoices' => 'Create Invoices',
        'record_payments' => 'Record Payments',
        'payment_reports' => 'Generate Payment Reports',
        'outstanding_balances' => 'View Outstanding Balances'
    ],
    'Financial Management' => [
        'view_revenue' => 'View Revenue Reports',
        'commission_management' => 'Manage Commissions',
        'credit_scoring' => 'Access Credit Scoring',
        'facility_loans' => 'Manage Facility Loans',
        'financial_reports' => 'Generate Financial Reports'
    ],
    'User Management' => [
        'view_users' => 'View Users',
        'add_users' => 'Add New Users',
        'edit_users' => 'Edit User Information',
        'manage_roles' => 'Manage User Roles',
        'user_permissions' => 'Manage User Permissions'
    ],
    'System Administration' => [
        'system_settings' => 'Modify System Settings',
        'backup_restore' => 'Backup & Restore Data',
        'activity_logs' => 'View Activity Logs',
        'system_maintenance' => 'System Maintenance',
        'api_access' => 'API Access Management'
    ],
    'Communications' => [
        'send_sms' => 'Send SMS Messages',
        'send_emails' => 'Send Email Messages',
        'bulk_messaging' => 'Bulk Messaging',
        'communication_templates' => 'Manage Communication Templates',
        'communication_logs' => 'View Communication Logs'
    ],
    'Reports & Analytics' => [
        'clinical_reports' => 'Clinical Reports',
        'financial_reports' => 'Financial Reports',
        'performance_reports' => 'Performance Reports',
        'custom_reports' => 'Custom Reports',
        'export_data' => 'Export Data'
    ]
];

// Default role permissions (based on the menu structure)
$default_role_permissions = [
    'Super Admin' => ['*'], // All permissions
    'Hospital Admin' => [
        'view_patients', 'add_patients', 'edit_patients', 'export_patients',
        'view_visits', 'register_visits', 'edit_visits', 'visit_reports',
        'view_consultations', 'consultation_reports', 'view_medical_history',
        'view_lab_orders', 'lab_reports',
        'view_invoices', 'create_invoices', 'record_payments', 'payment_reports', 'outstanding_balances',
        'view_revenue', 'financial_reports',
        'clinical_reports', 'financial_reports', 'performance_reports'
    ],
    'Doctor' => [
        'view_patients', 'edit_patients',
        'view_visits', 'visit_reports',
        'view_consultations', 'conduct_consultations', 'edit_consultations', 'view_medical_history',
        'view_lab_orders', 'create_lab_orders',
        'view_procedures', 'perform_procedures', 'schedule_procedures',
        'clinical_reports'
    ],
    'Senior Doctor' => [
        'view_patients', 'edit_patients',
        'view_visits', 'visit_reports',
        'view_consultations', 'conduct_consultations', 'edit_consultations', 'consultation_reports', 'view_medical_history',
        'view_lab_orders', 'create_lab_orders', 'approve_lab_results',
        'view_procedures', 'perform_procedures', 'schedule_procedures', 'procedure_reports',
        'clinical_reports', 'performance_reports'
    ],
    'Nurse' => [
        'view_patients', 'edit_patients',
        'view_visits', 'register_visits',
        'view_consultations',
        'view_procedures', 'perform_procedures',
        'record_payments'
    ],
    'Receptionist' => [
        'view_patients', 'add_patients', 'edit_patients',
        'view_visits', 'register_visits', 'edit_visits',
        'view_invoices', 'create_invoices', 'record_payments', 'outstanding_balances'
    ],
    'Lab Technician' => [
        'view_patients',
        'view_lab_orders', 'process_lab_tests', 'enter_lab_results',
        'lab_reports'
    ],
    'Lab Manager' => [
        'view_patients',
        'view_lab_orders', 'create_lab_orders', 'process_lab_tests', 'enter_lab_results', 'approve_lab_results',
        'lab_reports', 'performance_reports'
    ],
    'Billing Clerk' => [
        'view_patients',
        'view_visits',
        'view_invoices', 'create_invoices', 'record_payments', 'payment_reports', 'outstanding_balances',
        'financial_reports'
    ],
    'Pharmacist' => [
        'view_patients',
        'view_consultations'
    ],
    'Medical Director' => [
        'view_patients', 'add_patients', 'edit_patients', 'export_patients',
        'view_visits', 'register_visits', 'edit_visits', 'visit_reports',
        'view_consultations', 'conduct_consultations', 'edit_consultations', 'consultation_reports', 'view_medical_history',
        'view_lab_orders', 'create_lab_orders', 'approve_lab_results', 'lab_reports',
        'view_procedures', 'perform_procedures', 'schedule_procedures', 'procedure_reports',
        'view_revenue', 'commission_management', 'financial_reports',
        'view_users', 'add_users', 'edit_users',
        'clinical_reports', 'financial_reports', 'performance_reports', 'custom_reports'
    ],
    'Medical Records Officer' => [
        'view_patients', 'edit_patients', 'export_patients',
        'view_visits', 'visit_reports',
        'view_consultations', 'view_medical_history',
        'view_lab_orders', 'lab_reports',
        'view_procedures', 'procedure_reports',
        'clinical_reports', 'custom_reports', 'export_data'
    ]
];

// Get all roles
$roles_query = "SELECT role_id, role_name, description FROM user_roles WHERE is_active = 1 ORDER BY role_name";
$roles_result = $conn->query($roles_query);

// Get recent access attempts (simulated data for demonstration)
$recent_access = [
    ['user' => 'Dr. John Smith', 'action' => 'View Patient Records', 'status' => 'ALLOWED', 'time' => '2 minutes ago'],
    ['user' => 'Nurse Jane Doe', 'action' => 'Record Payment', 'status' => 'DENIED', 'time' => '5 minutes ago'],
    ['user' => 'Lab Tech Bob', 'action' => 'Enter Lab Results', 'status' => 'ALLOWED', 'time' => '8 minutes ago'],
    ['user' => 'Receptionist Mary', 'action' => 'Delete Patient Record', 'status' => 'DENIED', 'time' => '12 minutes ago'],
    ['user' => 'Dr. Sarah Wilson', 'action' => 'View Financial Reports', 'status' => 'DENIED', 'time' => '15 minutes ago']
];

// Get security statistics (simulated)
$security_stats = [
    'total_permissions' => array_sum(array_map('count', $system_modules)),
    'active_users_today' => 25,
    'failed_access_attempts' => 12,
    'successful_logins' => 156
];
?>

<?php include 'layouts/main.php'; ?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Access Control - Healthcare EMR System')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <style>
        /* Use the same styling from dashboard */
        body {
            background-color: #f8fafc;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        
        .page-content {
            padding: 1.5rem;
        }
        
        /* Page Header */
        .page-header {
            background: linear-gradient(135deg, #1e40af 0%, #3b82f6 100%);
            color: white;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .page-title {
            font-size: 1.75rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .page-subtitle {
            opacity: 0.9;
            margin: 0;
        }
        
        /* Stats Cards */
        .stats-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            text-align: center;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        
        .stat-label {
            font-size: 0.875rem;
            color: #64748b;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        /* Permission Matrix */
        .permission-matrix {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .matrix-header {
            padding: 1.5rem;
            background: #f8fafc;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .matrix-title {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
        }
        
        .matrix-body {
            padding: 0;
            max-height: 600px;
            overflow-y: auto;
        }
        
        .permission-table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
        }
        
        .permission-table th {
            background: #f8fafc;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            font-size: 0.875rem;
            color: #64748b;
            border-bottom: 1px solid #e2e8f0;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .permission-table td {
            padding: 0.75rem 1rem;
            border-bottom: 1px solid #f1f5f9;
            font-size: 0.875rem;
        }
        
        .permission-table tbody tr:hover {
            background: #f8fafc;
        }
        
        .module-header {
            background: #e2e8f0;
            font-weight: 600;
            color: #1e293b;
        }
        
        .permission-row {
            padding-left: 2rem;
        }
        
        .permission-check {
            text-align: center;
        }
        
        .permission-indicator {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: inline-block;
            margin: 0 auto;
        }
        
        .allowed { background: #059669; }
        .denied { background: #dc2626; }
        .partial { background: #d97706; }
        
        /* Role Tabs */
        .role-tabs {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            margin-bottom: 2rem;
        }
        
        .role-tabs-header {
            padding: 1.5rem 1.5rem 0;
        }
        
        .role-tabs-nav {
            display: flex;
            border-bottom: 1px solid #e2e8f0;
            overflow-x: auto;
            gap: 0.5rem;
        }
        
        .role-tab {
            padding: 0.75rem 1.5rem;
            border: none;
            background: transparent;
            color: #64748b;
            font-weight: 500;
            cursor: pointer;
            border-bottom: 2px solid transparent;
            white-space: nowrap;
            transition: all 0.2s ease;
        }
        
        .role-tab:hover {
            color: #3b82f6;
        }
        
        .role-tab.active {
            color: #3b82f6;
            border-bottom-color: #3b82f6;
        }
        
        .role-tab-content {
            display: none;
            padding: 1.5rem;
        }
        
        .role-tab-content.active {
            display: block;
        }
        
        /* Recent Access */
        .recent-access {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            margin-bottom: 2rem;
        }
        
        .access-header {
            padding: 1.5rem;
            background: #f8fafc;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .access-title {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
        }
        
        .access-item {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #f1f5f9;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .access-item:last-child {
            border-bottom: none;
        }
        
        .access-info {
            flex: 1;
        }
        
        .access-user {
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.25rem;
        }
        
        .access-action {
            color: #64748b;
            font-size: 0.875rem;
        }
        
        .access-status {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        .status-allowed { background: #dcfce7; color: #166534; }
        .status-denied { background: #fee2e2; color: #991b1b; }
        
        .access-time {
            color: #64748b;
            font-size: 0.75rem;
        }
        
        /* Security Alert */
        .security-alert {
            background: #fef3c7;
            border: 1px solid #f59e0b;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .security-alert-icon {
            width: 24px;
            height: 24px;
            background: #f59e0b;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 0.875rem;
        }
        
        .security-alert-content {
            flex: 1;
        }
        
        .security-alert-title {
            font-weight: 600;
            color: #92400e;
            margin-bottom: 0.25rem;
        }
        
        .security-alert-text {
            color: #92400e;
            font-size: 0.875rem;
        }
        
        /* Buttons */
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-primary {
            background: #3b82f6;
            color: white;
        }
        
        .btn-primary:hover {
            background: #1d4ed8;
            color: white;
        }
        
        .btn-warning {
            background: #d97706;
            color: white;
        }
        
        .btn-warning:hover {
            background: #b45309;
            color: white;
        }
        
        .btn-danger {
            background: #dc2626;
            color: white;
        }
        
        .btn-danger:hover {
            background: #b91c1c;
            color: white;
        }
        
        /* Legend */
        .permission-legend {
            display: flex;
            gap: 1rem;
            align-items: center;
            margin-bottom: 1rem;
            flex-wrap: wrap;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: #64748b;
        }
        
        .legend-indicator {
            width: 16px;
            height: 16px;
            border-radius: 50%;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .page-content {
                padding: 1rem;
            }
            
            .stats-row {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .role-tabs-nav {
                flex-wrap: wrap;
            }
            
            .permission-table {
                font-size: 0.75rem;
            }
            
            .permission-table th,
            .permission-table td {
                padding: 0.5rem;
            }
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <!-- Page Header -->
                    <div class="page-header">
                        <h1 class="page-title">Access Control & Permissions</h1>
                        <p class="page-subtitle">Manage user permissions and system security</p>
                    </div>

                    <!-- Security Alert -->
                    <div class="security-alert">
                        <div class="security-alert-icon">
                            <i class="ph-shield-warning"></i>
                        </div>
                        <div class="security-alert-content">
                            <div class="security-alert-title">Security Notice</div>
                            <div class="security-alert-text">
                                This is a demonstration of the access control system. In a production environment, 
                                permissions would be stored in the database and enforced throughout the application.
                            </div>
                        </div>
                    </div>

                    <!-- Stats Row -->
                    <div class="stats-row">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $security_stats['total_permissions']; ?></div>
                            <div class="stat-label">Total Permissions</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $security_stats['active_users_today']; ?></div>
                            <div class="stat-label">Active Users Today</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $security_stats['successful_logins']; ?></div>
                            <div class="stat-label">Successful Logins</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-value" style="color: #dc2626;"><?php echo $security_stats['failed_access_attempts']; ?></div>
                            <div class="stat-label">Failed Access Attempts</div>
                        </div>
                    </div>

                    <!-- Role-based Permissions Matrix -->
                    <div class="role-tabs">
                        <div class="role-tabs-header">
                            <h3 style="margin: 0 0 1rem 0; font-size: 1.125rem; font-weight: 600; color: #1e293b;">
                                Role-based Permission Matrix
                            </h3>
                            <div class="role-tabs-nav">
                                <?php 
                                $first_role = true;
                                foreach ($default_role_permissions as $role => $permissions): 
                                ?>
                                <button class="role-tab <?php echo $first_role ? 'active' : ''; ?>" 
                                        onclick="showRoleTab('<?php echo str_replace(' ', '_', strtolower($role)); ?>')">
                                    <?php echo htmlspecialchars($role); ?>
                                </button>
                                <?php 
                                $first_role = false;
                                endforeach; 
                                ?>
                            </div>
                        </div>
                        
                        <?php 
                        $first_role = true;
                        foreach ($default_role_permissions as $role => $role_permissions): 
                        ?>
                        <div id="<?php echo str_replace(' ', '_', strtolower($role)); ?>" 
                             class="role-tab-content <?php echo $first_role ? 'active' : ''; ?>">
                            
                            <div class="permission-legend">
                                <div class="legend-item">
                                    <div class="legend-indicator allowed"></div>
                                    <span>Full Access</span>
                                </div>
                                <div class="legend-item">
                                    <div class="legend-indicator partial"></div>
                                    <span>Partial Access</span>
                                </div>
                                <div class="legend-item">
                                    <div class="legend-indicator denied"></div>
                                    <span>No Access</span>
                                </div>
                            </div>

                            <div class="permission-matrix">
                                <div class="matrix-body">
                                    <table class="permission-table">
                                        <thead>
                                            <tr>
                                                <th style="width: 50%;">Module / Permission</th>
                                                <th style="width: 50%; text-align: center;">Access Level</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($system_modules as $module => $permissions): ?>
                                            <tr class="module-header">
                                                <td colspan="2"><?php echo htmlspecialchars($module); ?></td>
                                            </tr>
                                            <?php foreach ($permissions as $perm_key => $perm_name): ?>
                                            <tr>
                                                <td class="permission-row"><?php echo htmlspecialchars($perm_name); ?></td>
                                                <td class="permission-check">
                                                    <?php
                                                    $has_access = false;
                                                    if (in_array('*', $role_permissions)) {
                                                        $has_access = true;
                                                        $access_class = 'allowed';
                                                    } elseif (in_array($perm_key, $role_permissions)) {
                                                        $has_access = true;
                                                        $access_class = 'allowed';
                                                    } else {
                                                        $access_class = 'denied';
                                                    }
                                                    ?>
                                                    <div class="permission-indicator <?php echo $access_class; ?>" 
                                                         title="<?php echo $has_access ? 'Allowed' : 'Denied'; ?>"></div>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                        <?php 
                        $first_role = false;
                        endforeach; 
                        ?>
                    </div>

                    <!-- Recent Access Attempts -->
                    <div class="recent-access">
                        <div class="access-header">
                            <h3 class="access-title">Recent Access Attempts</h3>
                        </div>
                        <?php foreach ($recent_access as $access): ?>
                        <div class="access-item">
                            <div class="access-info">
                                <div class="access-user"><?php echo htmlspecialchars($access['user']); ?></div>
                                <div class="access-action"><?php echo htmlspecialchars($access['action']); ?></div>
                            </div>
                            <div class="access-status">
                                <span class="status-badge status-<?php echo strtolower($access['status']); ?>">
                                    <?php echo $access['status']; ?>
                                </span>
                                <div class="access-time"><?php echo $access['time']; ?></div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- Action Buttons -->
                    <div class="d-flex gap-3 justify-content-center">
                        <a href="user_roles.php" class="btn btn-primary">
                            <i class="ph-users me-1"></i> Manage User Roles
                        </a>
                        <a href="activity_logs.php" class="btn btn-warning">
                            <i class="ph-list-checks me-1"></i> View Activity Logs
                        </a>
                        <a href="security_logs.php" class="btn btn-danger">
                            <i class="ph-shield-warning me-1"></i> Security Logs
                        </a>
                    </div>

                    <!-- Implementation Note -->
                    <div style="background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 8px; padding: 1.5rem; margin-top: 2rem;">
                        <h4 style="color: #1e293b; margin-bottom: 1rem;">Implementation Notes</h4>
                        <div style="color: #64748b; font-size: 0.875rem; line-height: 1.6;">
                            <p><strong>This access control system demonstrates:</strong></p>
                            <ul>
                                <li>Role-based permission matrix showing what each role can access</li>
                                <li>Real-time monitoring of access attempts</li>
                                <li>Security statistics and alerts</li>
                                <li>Visual permission indicators for easy management</li>
                            </ul>
                            <p><strong>To implement in production:</strong></p>
                            <ul>
                                <li>Create a permissions table in the database</li>
                                <li>Implement middleware to check permissions on each page/action</li>
                                <li>Add audit logging for all access attempts</li>
                                <li>Implement role inheritance and custom permissions</li>
                                <li>Add two-factor authentication for sensitive operations</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        function showRoleTab(roleId) {
            // Hide all tab contents
            $('.role-tab-content').removeClass('active');
            $('.role-tab').removeClass('active');
            
            // Show selected tab content
            $('#' + roleId).addClass('active');
            $('[onclick*="' + roleId + '"]').addClass('active');
        }

        $(document).ready(function() {
            console.log('Access Control page loaded');
            
            // Add click handlers for permission indicators
            $('.permission-indicator').on('click', function() {
                const $this = $(this);
                const isAllowed = $this.hasClass('allowed');
                const permissionName = $this.closest('tr').find('.permission-row').text();
                const roleName = $('.role-tab.active').text();
                
                // This would integrate with a backend API to toggle permissions
                alert(`Permission: ${permissionName}\nRole: ${roleName}\nCurrent Status: ${isAllowed ? 'ALLOWED' : 'DENIED'}\n\nIn production, this would toggle the permission in the database.`);
            });
        });
    </script>

    <!-- App js -->
    <script src="assets/js/app.js"></script>
</body>
</html>

<?php $conn->close(); ?>