<?php
session_start();
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get claim ID from URL
$claim_no = isset($_GET['id']) ? $_GET['id'] : null;

if (!$claim_no) {
    header('Location: all_claims.php');
    exit();
}

// Fetch claim details with member and corporate info
$sql = "SELECT 
    b.*,
    m.surname,
    m.first_name,
    m.other_names,
    c.corporate,
    p.provider_name,
    bf.benefit as benefit_name,
    sv.service as service
FROM 
    bills b
    LEFT JOIN member_info m ON b.member_no = m.member_no
    LEFT JOIN corporate c ON b.corp_id = c.corp_id
    LEFT JOIN benefit bf ON b.claim_nature = bf.code
    LEFT JOIN services sv ON b.service = sv.code
    LEFT JOIN providers p ON b.provider = p.provider_id
WHERE 
    b.claim_no = ?";

$stmt = $conn->prepare($sql);
$stmt->bind_param("s", $claim_no);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header('Location: all_claims.php');
    exit();
}

// Get all claim lines
$claims = [];
$total_amount = 0;
while ($row = $result->fetch_assoc()) {
    $claims[] = $row;
    $total_amount += $row['invoiced_amount'];
}

// Get first row for header details
$claim_header = $claims[0];
?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'View Claim')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <style>
        .claim-status {
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-weight: 500;
        }
        .status-pending {
            background-color: #fff4de;
            color: #ffa500;
        }
        .status-approved {
            background-color: #e0f8e3;
            color: #2fb344;
        }
        .status-rejected {
            background-color: #ffe4e4;
            color: #ff0000;
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Claims', 'title' => 'View Claim')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header border-bottom-dashed">
                                    <div class="row g-4 align-items-center">
                                        <div class="col-sm">
                                            <div>
                                                <h5 class="card-title mb-0">Claim Details</h5>
                                            </div>
                                        </div>
                                        <div class="col-sm-auto">
                                            <div class="d-flex flex-wrap align-items-center gap-2">
                                                <button type="button" class="btn btn-success" onclick="window.print()">
                                                    <i class="ri-printer-line align-bottom me-1"></i> Print
                                                </button>
                                                <a href="all_claims.php" class="btn btn-primary">
                                                    <i class="ri-arrow-go-back-line align-bottom me-1"></i> Back
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="card-body">
                                    <div class="row g-3">
                                        <div class="col-lg-3 col-6">
                                            <p class="text-muted mb-2 text-uppercase fw-semibold">Claim No</p>
                                            <h5 class="fs-14 mb-0"><?php echo htmlspecialchars($claim_header['claim_no']); ?></h5>
                                        </div>
                                        <div class="col-lg-3 col-6">
                                            <p class="text-muted mb-2 text-uppercase fw-semibold">Date</p>
                                            <h5 class="fs-14 mb-0">
                                                <?php echo date('d M Y', strtotime($claim_header['invoice_date'])); ?>
                                            </h5>
                                        </div>
                                        <div class="col-lg-3 col-6">
                                            <p class="text-muted mb-2 text-uppercase fw-semibold">Status</p>
                                            <span class="claim-status <?php echo getStatusClass($claim_header['vet_status']); ?>">
                                                <?php echo getStatusText($claim_header['vet_status']); ?>
                                            </span>
                                        </div>
                                        <div class="col-lg-3 col-6">
                                            <p class="text-muted mb-2 text-uppercase fw-semibold">Total Amount</p>
                                            <h5 class="fs-14 mb-0">Ksh <?php echo number_format($total_amount, 2); ?></h5>
                                        </div>
                                    </div>
                                </div>
                                <div class="card-body p-4 border-top border-top-dashed">
                                    <div class="row g-3">
                                        <div class="col-6">
                                            <h6 class="text-muted text-uppercase fw-semibold mb-3">Member Information</h6>
                                            <p class="fw-medium mb-2"><?php 
                                                echo htmlspecialchars($claim_header['first_name'] . ' ' . 
                                                     $claim_header['surname']); 
                                            ?></p>
                                            <p class="text-muted mb-1">Member No: <?php 
                                                echo htmlspecialchars($claim_header['member_no']); 
                                            ?></p>
                                            <p class="text-muted mb-1">Corporate: <?php 
                                                echo htmlspecialchars($claim_header['corporate']); 
                                            ?></p>
                                        </div>
                                        <div class="col-6">
                                            <h6 class="text-muted text-uppercase fw-semibold mb-3">Provider Information</h6>
                                            <p class="fw-medium mb-2"><?php 
                                                echo htmlspecialchars($claim_header['provider_name']); 
                                            ?></p>
                                        </div>
                                    </div>
                                </div>
                                <div class="card-body p-4">
                                    <div class="table-responsive">
                                        <table class="table table-borderless text-center table-nowrap align-middle mb-0">
                                            <thead>
                                                <tr class="table-active">
                                                    <th scope="col" style="width: 50px;">#</th>
                                                    <th scope="col">Benefit</th>
                                                    <th scope="col">Service</th>
                                                    <th scope="col">Invoice No</th>
                                                    <th scope="col">Amount</th>
                                                    <th scope="col">Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($claims as $index => $claim): ?>
                                                <tr>
                                                    <td><?php echo $index + 1; ?></td>
                                                    <td><?php echo htmlspecialchars($claim['benefit_name']); ?></td>
                                                    <td><?php echo htmlspecialchars($claim['service']); ?></td>
                                                    <td><?php echo htmlspecialchars($claim['invoice_no']); ?></td>
                                                    <td>Ksh <?php echo number_format($claim['invoiced_amount'], 2); ?></td>
                                                    <td>
                                                        <span class="claim-status <?php 
                                                            echo getStatusClass($claim['vet_status']); 
                                                        ?>">
                                                            <?php echo getStatusText($claim['vet_status']); ?>
                                                        </span>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="assets/js/app.js"></script>
</body>
</html>

<?php
function getStatusClass($status) {
    if ($status === null) return 'status-pending';
    return $status ? 'status-approved' : 'status-rejected';
}

function getStatusText($status) {
    if ($status === null) return 'Pending';
    return $status ? 'Approved' : 'Rejected';
}

$conn->close();
?>