<?php
session_start();
include 'layouts/dbconn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get customer ID from URL
$customer_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$customer_id) {
    header('Location: list_customers.php');
    exit();
}

// Get customer details with related information
$sql = "SELECT 
    c.*,
    cs.segment_name,
    ra.full_name as referring_agent_name,
    ra.agent_code as referring_agent_code,
    kyc_verifier.full_name as kyc_verified_by_name
FROM customers c
LEFT JOIN customer_segments cs ON c.segment_id = cs.segment_id
LEFT JOIN users ra ON c.referring_agent_id = ra.user_id
LEFT JOIN users kyc_verifier ON c.kyc_verified_by = kyc_verifier.user_id
WHERE c.customer_id = ?";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $customer_id);
$stmt->execute();
$customer = $stmt->get_result()->fetch_assoc();

if (!$customer) {
    header('Location: list_customers.php?error=customer_not_found');
    exit();
}

// Get customer's policies
$policies_sql = "SELECT 
    p.*,
    ip.product_name,
    pc.category_name,
    ic.company_name,
    agent.full_name as agent_name,
    agent.agent_code
FROM policies p
LEFT JOIN insurance_products ip ON p.product_id = ip.product_id
LEFT JOIN product_categories pc ON ip.category_id = pc.category_id
LEFT JOIN insurance_companies ic ON p.company_id = ic.company_id
LEFT JOIN users agent ON p.agent_id = agent.user_id
WHERE p.customer_id = ?
ORDER BY p.created_at DESC";

$policies_stmt = $conn->prepare($policies_sql);
$policies_stmt->bind_param("i", $customer_id);
$policies_stmt->execute();
$policies = $policies_stmt->get_result();

// Get customer's claims
$claims_sql = "SELECT 
    cl.*,
    p.policy_number,
    ip.product_name
FROM claims cl
LEFT JOIN policies p ON cl.policy_id = p.policy_id
LEFT JOIN insurance_products ip ON p.product_id = ip.product_id
WHERE cl.customer_id = ?
ORDER BY cl.created_at DESC
LIMIT 10";

$claims_stmt = $conn->prepare($claims_sql);
$claims_stmt->bind_param("i", $customer_id);
$claims_stmt->execute();
$claims = $claims_stmt->get_result();

// Get customer's payments
$payments_sql = "SELECT 
    pp.*,
    p.policy_number,
    ip.product_name
FROM premium_payments pp
LEFT JOIN policies p ON pp.policy_id = p.policy_id
LEFT JOIN insurance_products ip ON p.product_id = ip.product_id
WHERE pp.customer_id = ?
ORDER BY pp.created_at DESC
LIMIT 10";

$payments_stmt = $conn->prepare($payments_sql);
$payments_stmt->bind_param("i", $customer_id);
$payments_stmt->execute();
$payments = $payments_stmt->get_result();

// Get customer's dependents
$dependents_sql = "SELECT * FROM customer_dependents WHERE customer_id = ? AND is_active = 1 ORDER BY created_at";
$dependents_stmt = $conn->prepare($dependents_sql);
$dependents_stmt->bind_param("i", $customer_id);
$dependents_stmt->execute();
$dependents = $dependents_stmt->get_result();

// Get customer statistics
$stats_sql = "SELECT 
    COUNT(DISTINCT p.policy_id) as total_policies,
    COUNT(DISTINCT CASE WHEN p.status = 'ACTIVE' THEN p.policy_id END) as active_policies,
    COUNT(DISTINCT cl.claim_id) as total_claims,
    COUNT(DISTINCT CASE WHEN cl.status = 'APPROVED' THEN cl.claim_id END) as approved_claims,
    COALESCE(SUM(DISTINCT pp.total_amount), 0) as total_premiums_paid,
    COALESCE(SUM(DISTINCT CASE WHEN pp.status = 'COMPLETED' THEN pp.total_amount END), 0) as completed_payments,
    COALESCE(SUM(DISTINCT cl.claimed_amount), 0) as total_claims_amount,
    COALESCE(SUM(DISTINCT CASE WHEN cl.status IN ('APPROVED', 'PAID') THEN cl.approved_amount END), 0) as total_claims_approved
FROM customers c
LEFT JOIN policies p ON c.customer_id = p.customer_id
LEFT JOIN claims cl ON c.customer_id = cl.customer_id
LEFT JOIN premium_payments pp ON c.customer_id = pp.customer_id
WHERE c.customer_id = ?";

$stats_stmt = $conn->prepare($stats_sql);
$stats_stmt->bind_param("i", $customer_id);
$stats_stmt->execute();
$stats = $stats_stmt->get_result()->fetch_assoc();

// Check for success message
$success_message = '';
if (isset($_GET['success'])) {
    $success_type = $_GET['success'];
    switch ($success_type) {
        case 'created':
            $success_message = 'Customer created successfully!';
            break;
        case 'updated':
            $success_message = 'Customer information updated successfully!';
            break;
        case 'kyc_verified':
            $success_message = 'Customer KYC verification completed!';
            break;
    }
}
?>

<?php include 'layouts/main.php'; ?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Customer Details - Ultrasure Insurance')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <style>
        .customer-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
        }
        
        .customer-avatar {
            width: 80px;
            height: 80px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            font-size: 2rem;
            font-weight: 700;
            color: white;
            margin-right: 20px;
        }
        
        .info-card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            margin-bottom: 20px;
        }
        
        .info-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }
        
        .info-card .card-header {
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
            border-radius: 15px 15px 0 0;
            padding: 15px 20px;
        }
        
        .info-card .card-header h6 {
            margin: 0;
            font-weight: 600;
            color: #495057;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px solid #f1f3f4;
        }
        
        .info-item:last-child {
            border-bottom: none;
        }
        
        .info-label {
            font-weight: 500;
            color: #6c757d;
            margin-bottom: 0;
        }
        
        .info-value {
            font-weight: 600;
            color: #495057;
            margin-bottom: 0;
            text-align: right;
        }
        
        .stats-card {
            border: none;
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .stats-card:hover {
            transform: translateY(-5px);
        }
        
        .stats-card.primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .stats-card.success {
            background: linear-gradient(135deg, #0ab39c 0%, #06b6d4 100%);
            color: white;
        }
        
        .stats-card.info {
            background: linear-gradient(135deg, #3577f1 0%, #7c3aed 100%);
            color: white;
        }
        
        .stats-card.warning {
            background: linear-gradient(135deg, #f7b84b 0%, #f59e0b 100%);
            color: white;
        }
        
        .stats-number {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 5px;
        }
        
        .stats-label {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .table-card {
            border: none;
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        }
        
        .table-card .card-header {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-bottom: 1px solid #dee2e6;
            padding: 15px 20px;
        }
        
        .table-enhanced thead th {
            background: #f8f9fa;
            border: none;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.8rem;
            letter-spacing: 0.5px;
            padding: 15px;
        }
        
        .table-enhanced tbody tr {
            transition: all 0.3s ease;
        }
        
        .table-enhanced tbody tr:hover {
            background-color: #f8f9fa;
        }
        
        .table-enhanced tbody td {
            padding: 15px;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .action-buttons .btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 500;
            margin-right: 8px;
            margin-bottom: 8px;
        }
        
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        
        .timeline::before {
            content: '';
            position: absolute;
            left: 15px;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #e9ecef;
        }
        
        .timeline-item {
            position: relative;
            margin-bottom: 20px;
        }
        
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -8px;
            top: 8px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #667eea;
            border: 3px solid white;
            box-shadow: 0 0 0 3px #e9ecef;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 500;
            padding: 10px 20px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .tab-content {
            margin-top: 20px;
        }
    </style>
    <style>
/* Add this to the existing CSS in view_customer.php */

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 20px; /* Increased padding from 12px 0 to 12px 20px */
    border-bottom: 1px solid #f1f3f4;
}

.info-item:last-child {
    border-bottom: none;
}

.info-label {
    font-weight: 500;
    color: #6c757d;
    margin-bottom: 0;
    min-width: 140px; /* Ensure consistent label width */
}

.info-value {
    font-weight: 600;
    color: #495057;
    margin-bottom: 0;
    text-align: right;
    flex: 1;
    margin-left: 15px; /* Add space between label and value */
}

.card-body {
    padding: 0; /* Remove default padding since we're adding it to info-items */
}

.card-header {
    padding: 15px 20px; /* Consistent header padding */
}
</style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    
                    <!-- Page Title -->
                    <div class="row">
                        <div class="col-12">
                            <div class="page-title-box d-sm-flex align-items-center justify-content-between">
                                <h4 class="mb-sm-0">Customer Details</h4>
                                <div class="page-title-right">
                                    <ol class="breadcrumb m-0">
                                        <li class="breadcrumb-item"><a href="index.php">Dashboard</a></li>
                                        <li class="breadcrumb-item"><a href="list_customers.php">Customers</a></li>
                                        <li class="breadcrumb-item active"><?php echo htmlspecialchars($customer['customer_number']); ?></li>
                                    </ol>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Success Message -->
                    <?php if($success_message): ?>
                        <div class="row">
                            <div class="col-12">
                                <div class="alert alert-success alert-dismissible fade show" role="alert">
                                    <i class="ri-check-line me-2"></i>
                                    <?php echo htmlspecialchars($success_message); ?>
                                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- Customer Header -->
                    <div class="customer-header">
                        <div class="d-flex align-items-center justify-content-between">
                            <div class="d-flex align-items-center">
                                <div class="customer-avatar">
                                    <?php echo strtoupper(substr($customer['first_name'], 0, 1) . substr($customer['last_name'], 0, 1)); ?>
                                </div>
                                <div>
                                    <h3 class="mb-1"><?php echo htmlspecialchars($customer['first_name'] . ' ' . $customer['last_name']); ?></h3>
                                    <p class="mb-1 opacity-75">Customer #<?php echo htmlspecialchars($customer['customer_number']); ?></p>
                                    <div class="d-flex align-items-center gap-3">
                                        <?php if($customer['email']): ?>
                                            <span><i class="ri-mail-line me-1"></i> <?php echo htmlspecialchars($customer['email']); ?></span>
                                        <?php endif; ?>
                                        <span><i class="ri-phone-line me-1"></i> <?php echo htmlspecialchars($customer['phone_primary']); ?></span>
                                        <?php if($customer['segment_name']): ?>
                                            <span class="badge bg-light text-dark"><i class="ri-user-line me-1"></i> <?php echo htmlspecialchars($customer['segment_name']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="action-buttons">
                                <a href="edit_customer.php?id=<?php echo $customer['customer_id']; ?>" class="btn btn-light">
                                    <i class="ri-edit-line me-1"></i> Edit Customer
                                </a>
                                <a href="new_quote.php?customer_id=<?php echo $customer['customer_id']; ?>" class="btn btn-warning">
                                    <i class="ri-file-add-line me-1"></i> Generate Quote
                                </a>
                                <?php if($customer['kyc_status'] !== 'VERIFIED'): ?>
                                    <button class="btn btn-success" onclick="verifyKYC(<?php echo $customer['customer_id']; ?>)">
                                        <i class="ri-shield-check-line me-1"></i> Verify KYC
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Statistics Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-3 col-md-6">
                            <div class="stats-card primary">
                                <div class="stats-number"><?php echo $stats['total_policies']; ?></div>
                                <div class="stats-label">Total Policies</div>
                                <small class="opacity-75"><?php echo $stats['active_policies']; ?> Active</small>
                            </div>
                        </div>
                        
                        <div class="col-xl-3 col-md-6">
                            <div class="stats-card success">
                                <div class="stats-number">KSH <?php echo number_format($stats['completed_payments'], 0); ?></div>
                                <div class="stats-label">Premiums Paid</div>
                                <small class="opacity-75">Total: KSH <?php echo number_format($stats['total_premiums_paid'], 0); ?></small>
                            </div>
                        </div>
                        
                        <div class="col-xl-3 col-md-6">
                            <div class="stats-card info">
                                <div class="stats-number"><?php echo $stats['total_claims']; ?></div>
                                <div class="stats-label">Total Claims</div>
                                <small class="opacity-75"><?php echo $stats['approved_claims']; ?> Approved</small>
                            </div>
                        </div>
                        
                        <div class="col-xl-3 col-md-6">
                            <div class="stats-card warning">
                                <div class="stats-number">KSH <?php echo number_format($stats['total_claims_approved'], 0); ?></div>
                                <div class="stats-label">Claims Approved</div>
                                <small class="opacity-75">Total: KSH <?php echo number_format($stats['total_claims_amount'], 0); ?></small>
                            </div>
                        </div>
                    </div>

                    <!-- Main Content Tabs -->
                    <div class="row">
                        <div class="col-12">
                            <ul class="nav nav-pills nav-justified mb-4" id="customerTabs" role="tablist">
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link active" id="overview-tab" data-bs-toggle="pill" data-bs-target="#overview" type="button" role="tab">
                                        <i class="ri-user-line me-2"></i>Overview
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="policies-tab" data-bs-toggle="pill" data-bs-target="#policies" type="button" role="tab">
                                        <i class="ri-shield-check-line me-2"></i>Policies (<?php echo $stats['total_policies']; ?>)
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="claims-tab" data-bs-toggle="pill" data-bs-target="#claims" type="button" role="tab">
                                        <i class="ri-hand-heart-line me-2"></i>Claims (<?php echo $stats['total_claims']; ?>)
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="payments-tab" data-bs-toggle="pill" data-bs-target="#payments" type="button" role="tab">
                                        <i class="ri-money-dollar-circle-line me-2"></i>Payments
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="dependents-tab" data-bs-toggle="pill" data-bs-target="#dependents" type="button" role="tab">
                                        <i class="ri-group-line me-2"></i>Dependents (<?php echo $dependents->num_rows; ?>)
                                    </button>
                                </li>
                            </ul>
                            
                            <div class="tab-content" id="customerTabsContent">
                                
                                <!-- Overview Tab -->
                                <div class="tab-pane fade show active" id="overview" role="tabpanel">
                                    <div class="row">
                                        <!-- Personal Information -->
                                        <div class="col-xl-6 col-lg-6">
                                            <div class="info-card">
                                                <div class="card-header">
                                                    <h6><i class="ri-user-line me-2"></i>Personal Information</h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="info-item">
                                                        <p class="info-label">Full Name</p>
                                                        <p class="info-value"><?php echo htmlspecialchars($customer['first_name'] . ' ' . ($customer['middle_name'] ? $customer['middle_name'] . ' ' : '') . $customer['last_name']); ?></p>
                                                    </div>
                                                    <?php if($customer['date_of_birth']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Date of Birth</p>
                                                        <p class="info-value">
                                                            <?php 
                                                            $dob = new DateTime($customer['date_of_birth']);
                                                            $today = new DateTime();
                                                            $age = $today->diff($dob)->y;
                                                            echo $dob->format('M j, Y') . ' (' . $age . ' years)';
                                                            ?>
                                                        </p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if($customer['gender']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Gender</p>
                                                        <p class="info-value"><?php echo ucfirst(strtolower($customer['gender'])); ?></p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if($customer['marital_status']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Marital Status</p>
                                                        <p class="info-value"><?php echo ucfirst(strtolower($customer['marital_status'])); ?></p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if($customer['national_id']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">National ID</p>
                                                        <p class="info-value"><?php echo htmlspecialchars($customer['national_id']); ?></p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Customer Type</p>
                                                        <p class="info-value"><?php echo ucfirst(strtolower($customer['customer_type'])); ?></p>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Contact Information -->
                                        <div class="col-xl-6 col-lg-6">
                                            <div class="info-card">
                                                <div class="card-header">
                                                    <h6><i class="ri-phone-line me-2"></i>Contact Information</h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="info-item">
                                                        <p class="info-label">Primary Phone</p>
                                                        <p class="info-value">
                                                            <a href="tel:<?php echo $customer['phone_primary']; ?>" class="text-decoration-none">
                                                                <?php echo htmlspecialchars($customer['phone_primary']); ?>
                                                            </a>
                                                        </p>
                                                    </div>
                                                    <?php if($customer['phone_secondary']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Secondary Phone</p>
                                                        <p class="info-value">
                                                            <a href="tel:<?php echo $customer['phone_secondary']; ?>" class="text-decoration-none">
                                                                <?php echo htmlspecialchars($customer['phone_secondary']); ?>
                                                            </a>
                                                        </p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if($customer['email']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Email</p>
                                                        <p class="info-value">
                                                            <a href="mailto:<?php echo $customer['email']; ?>" class="text-decoration-none">
                                                                <?php echo htmlspecialchars($customer['email']); ?>
                                                            </a>
                                                        </p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Preferred Language</p>
                                                        <p class="info-value"><?php echo $customer['preferred_language'] == 'en' ? 'English' : 'Swahili'; ?></p>
                                                    </div>
                                                    <div class="info-item">
                                                        <p class="info-label">Communication Preference</p>
                                                        <p class="info-value"><?php echo ucfirst(strtolower($customer['communication_preference'])); ?></p>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Address Information -->
                                        <div class="col-xl-6 col-lg-6">
                                            <div class="info-card">
                                                <div class="card-header">
                                                    <h6><i class="ri-map-pin-line me-2"></i>Address Information</h6>
                                                </div>
                                                <div class="card-body">
                                                    <?php if($customer['address_line1'] || $customer['address_line2'] || $customer['city'] || $customer['county']): ?>
                                                        <div class="info-item">
                                                            <p class="info-label">Address</p>
                                                            <p class="info-value">
                                                                <?php 
                                                                $address_parts = array_filter([
                                                                    $customer['address_line1'],
                                                                    $customer['address_line2'],
                                                                    $customer['city'],
                                                                    $customer['county'],
                                                                    $customer['postal_code'],
                                                                    $customer['country']
                                                                ]);
                                                                echo htmlspecialchars(implode(', ', $address_parts));
                                                                ?>
                                                            </p>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="text-center py-3">
                                                            <i class="ri-map-pin-line fs-1 text-muted"></i>
                                                            <p class="text-muted mb-0">No address information available</p>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Account Status & KYC -->
                                        <div class="col-xl-6 col-lg-6">
                                            <div class="info-card">
                                                <div class="card-header">
                                                    <h6><i class="ri-shield-check-line me-2"></i>Account Status</h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="info-item">
                                                        <p class="info-label">Account Status</p>
                                                        <p class="info-value">
                                                            <?php if($customer['is_active']): ?>
                                                                <span class="status-badge bg-success text-white">Active</span>
                                                            <?php else: ?>
                                                                <span class="status-badge bg-danger text-white">Inactive</span>
                                                            <?php endif; ?>
                                                        </p>
                                                    </div>
                                                    <div class="info-item">
                                                        <p class="info-label">KYC Status</p>
                                                        <p class="info-value">
                                                            <?php
                                                            $kyc_class = match($customer['kyc_status']) {
                                                                'VERIFIED' => 'success',
                                                                'PENDING' => 'warning',
                                                                'REJECTED' => 'danger',
                                                                default => 'secondary'
                                                            };
                                                            ?>
                                                            <span class="status-badge bg-<?php echo $kyc_class; ?> text-white">
                                                                <?php echo ucfirst(strtolower($customer['kyc_status'])); ?>
                                                            </span>
                                                        </p>
                                                    </div>
                                                    <?php if($customer['kyc_verified_at'] && $customer['kyc_verified_by_name']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">KYC Verified</p>
                                                        <p class="info-value">
                                                            <?php echo date('M j, Y', strtotime($customer['kyc_verified_at'])); ?><br>
                                                            <small class="text-muted">by <?php echo htmlspecialchars($customer['kyc_verified_by_name']); ?></small>
                                                        </p>
                                                    </div>
                                                    <?php endif; ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Customer Since</p>
                                                        <p class="info-value"><?php echo date('M j, Y', strtotime($customer['created_at'])); ?></p>
                                                    </div>
                                                    <?php if($customer['referring_agent_name']): ?>
                                                    <div class="info-item">
                                                        <p class="info-label">Referring Agent</p>
                                                        <p class="info-value">
                                                            <?php echo htmlspecialchars($customer['referring_agent_name']); ?>
                                                            <small class="text-muted">(<?php echo htmlspecialchars($customer['referring_agent_code']); ?>)</small>
                                                        </p>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Policies Tab -->
                                <div class="tab-pane fade" id="policies" role="tabpanel">
                                    <div class="table-card">
                                        <div class="card-header">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <h6 class="mb-0">Customer Policies</h6>
                                                <a href="quotes/new_quote.php?customer_id=<?php echo $customer['customer_id']; ?>" class="btn btn-primary btn-sm">
                                                    <i class="ri-add-line me-1"></i> Generate Quote
                                                </a>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <?php if($policies->num_rows > 0): ?>
                                                <div class="table-responsive">
                                                    <table class="table table-enhanced mb-0">
                                                        <thead>
                                                            <tr>
                                                                <th>Policy Details</th>
                                                                <th>Product</th>
                                                                <th>Coverage</th>
                                                                <th>Premium</th>
                                                                <th>Status</th>
                                                                <th>Agent</th>
                                                                <th>Actions</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php while($policy = $policies->fetch_assoc()): ?>
                                                            <tr>
                                                                <td>
                                                                    <div>
                                                                        <h6 class="mb-1"><?php echo htmlspecialchars($policy['policy_number']); ?></h6>
                                                                        <small class="text-muted">
                                                                            <?php echo date('M j, Y', strtotime($policy['inception_date'])); ?> - 
                                                                            <?php echo date('M j, Y', strtotime($policy['expiry_date'])); ?>
                                                                        </small>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <div>
                                                                        <span class="fw-medium"><?php echo htmlspecialchars($policy['product_name']); ?></span><br>
                                                                        <small class="text-muted"><?php echo htmlspecialchars($policy['category_name']); ?></small><br>
                                                                        <small class="text-muted"><?php echo htmlspecialchars($policy['company_name']); ?></small>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <span class="fw-medium">KSH <?php echo number_format($policy['sum_assured'], 2); ?></span>
                                                                </td>
                                                                <td>
                                                                    <span class="fw-medium text-success">KSH <?php echo number_format($policy['premium_amount'], 2); ?></span><br>
                                                                    <small class="text-muted"><?php echo ucfirst(strtolower($policy['premium_frequency'])); ?></small>
                                                                </td>
                                                                <td>
                                                                    <?php
                                                                    $status_class = match($policy['status']) {
                                                                        'ACTIVE' => 'success',
                                                                        'PENDING' => 'warning',
                                                                        'LAPSED' => 'danger',
                                                                        'CANCELLED' => 'secondary',
                                                                        'EXPIRED' => 'dark',
                                                                        default => 'info'
                                                                    };
                                                                    ?>
                                                                    <span class="status-badge bg-<?php echo $status_class; ?> text-white">
                                                                        <?php echo ucfirst(strtolower($policy['status'])); ?>
                                                                    </span>
                                                                </td>
                                                                <td>
                                                                    <small class="text-muted">
                                                                        <?php echo htmlspecialchars($policy['agent_name']); ?><br>
                                                                        <?php echo htmlspecialchars($policy['agent_code']); ?>
                                                                    </small>
                                                                </td>
                                                                <td>
                                                                    <div class="dropdown">
                                                                        <button class="btn btn-soft-secondary btn-sm dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                                            Actions
                                                                        </button>
                                                                        <ul class="dropdown-menu">
                                                                            <li><a class="dropdown-item" href="policies/view_policy.php?id=<?php echo $policy['policy_id']; ?>">View Details</a></li>
                                                                            <li><a class="dropdown-item" href="policies/policy_documents.php?id=<?php echo $policy['policy_id']; ?>">Documents</a></li>
                                                                            <?php if($policy['status'] == 'ACTIVE'): ?>
                                                                                <li><a class="dropdown-item" href="claims/submit_claim.php?policy_id=<?php echo $policy['policy_id']; ?>">Submit Claim</a></li>
                                                                                <li><a class="dropdown-item" href="payments/record_payment.php?policy_id=<?php echo $policy['policy_id']; ?>">Record Payment</a></li>
                                                                            <?php endif; ?>
                                                                        </ul>
                                                                    </div>
                                                                </td>
                                                            </tr>
                                                            <?php endwhile; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center py-5">
                                                    <i class="ri-shield-line fs-1 text-muted"></i>
                                                    <h5 class="text-muted mt-3">No Policies Found</h5>
                                                    <p class="text-muted">This customer doesn't have any policies yet.</p>
                                                    <a href="quotes/new_quote.php?customer_id=<?php echo $customer['customer_id']; ?>" class="btn btn-primary">
                                                        <i class="ri-file-add-line me-1"></i> Generate First Quote
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Claims Tab -->
                                <div class="tab-pane fade" id="claims" role="tabpanel">
                                    <div class="table-card">
                                        <div class="card-header">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <h6 class="mb-0">Customer Claims</h6>
                                                <a href="claims/submit_claim.php?customer_id=<?php echo $customer['customer_id']; ?>" class="btn btn-primary btn-sm">
                                                    <i class="ri-add-line me-1"></i> Submit Claim
                                                </a>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <?php if($claims->num_rows > 0): ?>
                                                <div class="table-responsive">
                                                    <table class="table table-enhanced mb-0">
                                                        <thead>
                                                            <tr>
                                                                <th>Claim Details</th>
                                                                <th>Policy</th>
                                                                <th>Incident Date</th>
                                                                <th>Claimed Amount</th>
                                                                <th>Status</th>
                                                                <th>Actions</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php while($claim = $claims->fetch_assoc()): ?>
                                                            <tr>
                                                                <td>
                                                                    <div>
                                                                        <h6 class="mb-1"><?php echo htmlspecialchars($claim['claim_number']); ?></h6>
                                                                        <small class="text-muted">
                                                                            <?php echo ucfirst(strtolower($claim['claim_type'])); ?>
                                                                        </small>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <div>
                                                                        <span class="fw-medium"><?php echo htmlspecialchars($claim['policy_number']); ?></span><br>
                                                                        <small class="text-muted"><?php echo htmlspecialchars($claim['product_name']); ?></small>
                                                                    </div>
                                                                </td>
                                                                <td><?php echo date('M j, Y', strtotime($claim['incident_date'])); ?></td>
                                                                <td>
                                                                    <span class="fw-medium">KSH <?php echo number_format($claim['claimed_amount'], 2); ?></span>
                                                                    <?php if($claim['approved_amount']): ?>
                                                                        <br><small class="text-success">Approved: KSH <?php echo number_format($claim['approved_amount'], 2); ?></small>
                                                                    <?php endif; ?>
                                                                </td>
                                                                <td>
                                                                    <?php
                                                                    $claim_status_class = match($claim['status']) {
                                                                        'SUBMITTED' => 'info',
                                                                        'UNDER_INVESTIGATION' => 'warning',
                                                                        'APPROVED' => 'success',
                                                                        'REJECTED' => 'danger',
                                                                        'PAID' => 'success',
                                                                        default => 'secondary'
                                                                    };
                                                                    ?>
                                                                    <span class="status-badge bg-<?php echo $claim_status_class; ?> text-white">
                                                                        <?php echo ucfirst(str_replace('_', ' ', strtolower($claim['status']))); ?>
                                                                    </span>
                                                                </td>
                                                                <td>
                                                                    <a href="claims/view_claim.php?id=<?php echo $claim['claim_id']; ?>" class="btn btn-soft-primary btn-sm">
                                                                        <i class="ri-eye-line"></i>
                                                                    </a>
                                                                </td>
                                                            </tr>
                                                            <?php endwhile; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center py-5">
                                                    <i class="ri-hand-heart-line fs-1 text-muted"></i>
                                                    <h5 class="text-muted mt-3">No Claims Found</h5>
                                                    <p class="text-muted">This customer hasn't submitted any claims yet.</p>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Payments Tab -->
                                <div class="tab-pane fade" id="payments" role="tabpanel">
                                    <div class="table-card">
                                        <div class="card-header">
                                            <h6 class="mb-0">Payment History</h6>
                                        </div>
                                        <div class="card-body p-0">
                                            <?php if($payments->num_rows > 0): ?>
                                                <div class="table-responsive">
                                                    <table class="table table-enhanced mb-0">
                                                        <thead>
                                                            <tr>
                                                                <th>Payment Details</th>
                                                                <th>Policy</th>
                                                                <th>Amount</th>
                                                                <th>Payment Method</th>
                                                                <th>Status</th>
                                                                <th>Date</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php while($payment = $payments->fetch_assoc()): ?>
                                                            <tr>
                                                                <td>
                                                                    <div>
                                                                        <h6 class="mb-1"><?php echo htmlspecialchars($payment['payment_number']); ?></h6>
                                                                        <small class="text-muted">
                                                                            <?php echo ucfirst(strtolower($payment['payment_type'])); ?>
                                                                        </small>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <div>
                                                                        <span class="fw-medium"><?php echo htmlspecialchars($payment['policy_number']); ?></span><br>
                                                                        <small class="text-muted"><?php echo htmlspecialchars($payment['product_name']); ?></small>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <span class="fw-medium text-success">KSH <?php echo number_format($payment['total_amount'], 2); ?></span>
                                                                    <?php if($payment['premium_amount'] != $payment['total_amount']): ?>
                                                                        <br><small class="text-muted">Premium: KSH <?php echo number_format($payment['premium_amount'], 2); ?></small>
                                                                    <?php endif; ?>
                                                                </td>
                                                                <td><?php echo ucfirst(str_replace('_', ' ', strtolower($payment['payment_method']))); ?></td>
                                                                <td>
                                                                    <?php
                                                                    $payment_status_class = match($payment['status']) {
                                                                        'COMPLETED' => 'success',
                                                                        'PENDING' => 'warning',
                                                                        'FAILED' => 'danger',
                                                                        'REFUNDED' => 'info',
                                                                        default => 'secondary'
                                                                    };
                                                                    ?>
                                                                    <span class="status-badge bg-<?php echo $payment_status_class; ?> text-white">
                                                                        <?php echo ucfirst(strtolower($payment['status'])); ?>
                                                                    </span>
                                                                </td>
                                                                <td><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></td>
                                                            </tr>
                                                            <?php endwhile; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center py-5">
                                                    <i class="ri-money-dollar-circle-line fs-1 text-muted"></i>
                                                    <h5 class="text-muted mt-3">No Payments Found</h5>
                                                    <p class="text-muted">This customer hasn't made any payments yet.</p>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Dependents Tab -->
                                <div class="tab-pane fade" id="dependents" role="tabpanel">
                                    <div class="table-card">
                                        <div class="card-header">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <h6 class="mb-0">Customer Dependents</h6>
                                                <button class="btn btn-primary btn-sm" onclick="addDependent()">
                                                    <i class="ri-add-line me-1"></i> Add Dependent
                                                </button>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <?php if($dependents->num_rows > 0): ?>
                                                <div class="table-responsive">
                                                    <table class="table table-enhanced mb-0">
                                                        <thead>
                                                            <tr>
                                                                <th>Name</th>
                                                                <th>Relationship</th>
                                                                <th>Date of Birth</th>
                                                                <th>Age</th>
                                                                <th>Gender</th>
                                                                <th>Contact</th>
                                                                <th>Actions</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php while($dependent = $dependents->fetch_assoc()): ?>
                                                            <tr>
                                                                <td>
                                                                    <div class="d-flex align-items-center">
                                                                        <div class="customer-avatar bg-soft-info text-info me-3" style="width: 35px; height: 35px; font-size: 0.9rem;">
                                                                            <?php echo strtoupper(substr($dependent['first_name'], 0, 1) . substr($dependent['last_name'], 0, 1)); ?>
                                                                        </div>
                                                                        <div>
                                                                            <h6 class="mb-0"><?php echo htmlspecialchars($dependent['first_name'] . ' ' . $dependent['last_name']); ?></h6>
                                                                            <?php if($dependent['national_id']): ?>
                                                                                <small class="text-muted">ID: <?php echo htmlspecialchars($dependent['national_id']); ?></small>
                                                                            <?php endif; ?>
                                                                        </div>
                                                                    </div>
                                                                </td>
                                                                <td>
                                                                    <span class="badge bg-soft-primary text-primary">
                                                                        <?php echo ucfirst(strtolower($dependent['relationship'])); ?>
                                                                    </span>
                                                                </td>
                                                                <td><?php echo date('M j, Y', strtotime($dependent['date_of_birth'])); ?></td>
                                                                <td>
                                                                    <?php 
                                                                    $dob = new DateTime($dependent['date_of_birth']);
                                                                    $today = new DateTime();
                                                                    $age = $today->diff($dob)->y;
                                                                    echo $age . ' years';
                                                                    ?>
                                                                </td>
                                                                <td><?php echo ucfirst(strtolower($dependent['gender'])); ?></td>
                                                                <td>
                                                                    <?php if($dependent['phone']): ?>
                                                                        <a href="tel:<?php echo $dependent['phone']; ?>" class="text-decoration-none">
                                                                            <?php echo htmlspecialchars($dependent['phone']); ?>
                                                                        </a>
                                                                    <?php else: ?>
                                                                        <span class="text-muted">-</span>
                                                                    <?php endif; ?>
                                                                </td>
                                                                <td>
                                                                    <div class="dropdown">
                                                                        <button class="btn btn-soft-secondary btn-sm dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                                            Actions
                                                                        </button>
                                                                        <ul class="dropdown-menu">
                                                                            <li><a class="dropdown-item" href="#" onclick="editDependent(<?php echo $dependent['dependent_id']; ?>)">Edit</a></li>
                                                                            <li><a class="dropdown-item text-danger" href="#" onclick="removeDependent(<?php echo $dependent['dependent_id']; ?>)">Remove</a></li>
                                                                        </ul>
                                                                    </div>
                                                                </td>
                                                            </tr>
                                                            <?php endwhile; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center py-5">
                                                    <i class="ri-group-line fs-1 text-muted"></i>
                                                    <h5 class="text-muted mt-3">No Dependents Added</h5>
                                                    <p class="text-muted">Add family members or dependents for family insurance policies.</p>
                                                    <button class="btn btn-primary" onclick="addDependent()">
                                                        <i class="ri-add-line me-1"></i> Add First Dependent
                                                    </button>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                            </div>
                        </div>
                    </div>

                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <!-- Add Dependent Modal -->
    <div class="modal fade" id="addDependentModal" tabindex="-1" aria-labelledby="addDependentModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addDependentModalLabel">Add Dependent</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="addDependentForm">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">First Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="first_name" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Last Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="last_name" required>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Date of Birth <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" name="date_of_birth" required>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Gender <span class="text-danger">*</span></label>
                                <select class="form-select" name="gender" required>
                                    <option value="">Select Gender</option>
                                    <option value="MALE">Male</option>
                                    <option value="FEMALE">Female</option>
                                </select>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Relationship <span class="text-danger">*</span></label>
                                <select class="form-select" name="relationship" required>
                                    <option value="">Select Relationship</option>
                                    <option value="SPOUSE">Spouse</option>
                                    <option value="CHILD">Child</option>
                                    <option value="PARENT">Parent</option>
                                    <option value="SIBLING">Sibling</option>
                                    <option value="OTHER">Other</option>
                                </select>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">National ID</label>
                                <input type="text" class="form-control" name="national_id" maxlength="8">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Phone Number</label>
                                <input type="tel" class="form-control" name="phone">
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Add Dependent</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <script>
        function verifyKYC(customerId) {
    Swal.fire({
        title: 'Verify Customer KYC',
        text: 'Are you sure you want to verify this customer\'s KYC status?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, Verify KYC',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            // Show loading
            Swal.fire({
                title: 'Verifying KYC...',
                text: 'Please wait while we process the verification.',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            // Make AJAX request to verify KYC
            fetch('verify_kyc.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'customer_id=' + encodeURIComponent(customerId)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    Swal.fire({
                        title: 'Success!',
                        text: data.message || 'Customer KYC has been verified successfully.',
                        icon: 'success',
                        confirmButtonColor: '#28a745'
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire({
                        title: 'Error!',
                        text: data.message || 'Failed to verify KYC. Please try again.',
                        icon: 'error',
                        confirmButtonColor: '#dc3545'
                    });
                }
            })
            .catch(error => {
                console.error('Error:', error);
                Swal.fire({
                    title: 'Error!',
                    text: 'An error occurred while processing the request. Please try again.',
                    icon: 'error',
                    confirmButtonColor: '#dc3545'
                });
            });
        }
    });
}
        
        function addDependent() {
            const modal = new bootstrap.Modal(document.getElementById('addDependentModal'));
            modal.show();
        }
        
        function editDependent(dependentId) {
            // Implementation for editing dependent
            console.log('Edit dependent:', dependentId);
        }
        
        function removeDependent(dependentId) {
            Swal.fire({
                title: 'Remove Dependent',
                text: 'Are you sure you want to remove this dependent?',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc3545',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'Yes, Remove'
            }).then((result) => {
                if (result.isConfirmed) {
                    // Make AJAX request to remove dependent
                    fetch('remove_dependent.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'dependent_id=' + dependentId
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                title: 'Removed!',
                                text: 'Dependent has been removed successfully.',
                                icon: 'success'
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                title: 'Error!',
                                text: data.message || 'Failed to remove dependent.',
                                icon: 'error'
                            });
                        }
                    });
                }
            });
        }
        
        // Handle add dependent form submission
        document.getElementById('addDependentForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('customer_id', <?php echo $customer_id; ?>);
            
            fetch('add_dependent.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    bootstrap.Modal.getInstance(document.getElementById('addDependentModal')).hide();
                    Swal.fire({
                        title: 'Success!',
                        text: 'Dependent added successfully.',
                        icon: 'success'
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire({
                        title: 'Error!',
                        text: data.message || 'Failed to add dependent.',
                        icon: 'error'
                    });
                }
            })
            .catch(error => {
                Swal.fire({
                    title: 'Error!',
                    text: 'An error occurred. Please try again.',
                    icon: 'error'
                });
            });
        });
        
        // Auto-hide success alerts
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert-success');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);
        
        // Initialize tooltips
        const tooltips = document.querySelectorAll('[data-bs-toggle="tooltip"]');
        tooltips.forEach(tooltip => {
            new bootstrap.Tooltip(tooltip);
        });
    </script>
</body>
</html>

<?php
$stmt->close();
$policies_stmt->close();
$claims_stmt->close();
$payments_stmt->close();
$dependents_stmt->close();
$stats_stmt->close();
$conn->close();
?>