<?php
session_start();
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get quotation number from URL
$quotation_no = $_GET['no'] ?? '';
if (empty($quotation_no)) {
    header('Location: quotations_list.php');
    exit();
}

// Fetch quotation details with corporate info
$sql = "SELECT q.*, c.corporate as corporate_name, c.scheme
        FROM quotations q
        LEFT JOIN corporate c ON q.corp_id = c.corp_id
        WHERE q.quotation_no = ?";

$stmt = $conn->prepare($sql);
$stmt->bind_param('s', $quotation_no);
$stmt->execute();
$quotation = $stmt->get_result()->fetch_assoc();

if (!$quotation) {
    header('Location: quotations_list.php');
    exit();
}

// Fetch quotation members with their benefits
$sql = "SELECT qm.*, 
               qmb.benefit_code,
               qmb.premium as benefit_premium,
               qmb.limit_amount,
               b.benefit as benefit_name
        FROM quotation_members qm
        LEFT JOIN quotation_member_benefits qmb ON qm.member_id = qmb.member_id
        LEFT JOIN benefit b ON qmb.benefit_code = b.code
        WHERE qm.quotation_no = ?
        ORDER BY qm.member_id, qmb.benefit_code";

$stmt = $conn->prepare($sql);
$stmt->bind_param('s', $quotation_no);
$stmt->execute();
$result = $stmt->get_result();

$members = [];
while ($row = $result->fetch_assoc()) {
    $member_id = $row['member_id'];
    if (!isset($members[$member_id])) {
        $members[$member_id] = [
            'relation' => $row['relation'],
            'age' => $row['age'],
            'premium' => $row['premium'],
            'benefits' => []
        ];
    }
    if ($row['benefit_code']) {
        $members[$member_id]['benefits'][] = [
            'code' => $row['benefit_code'],
            'name' => $row['benefit_name'],
            'premium' => $row['benefit_premium'],
            'limit' => $row['limit_amount']
        ];
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'View Quotation')); ?>
    <?php include 'layouts/head-css.php'; ?>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Quotations', 'title' => 'View Quotation')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header align-items-center d-flex">
                                    <h4 class="card-title mb-0 flex-grow-1">Quotation Details</h4>
                                    <div class="flex-shrink-0">
                                        <div class="d-flex gap-2">
                                            <a href="download_quotation.php?no=<?php echo $quotation_no; ?>" 
                                               class="btn btn-info btn-sm">
                                                <i class="ri-download-2-line align-bottom"></i> Download
                                            </a>
                                            <?php if ($quotation['status'] === 'PENDING'): ?>
                                            <a href="edit_quotation.php?no=<?php echo $quotation_no; ?>" 
                                               class="btn btn-primary btn-sm">
                                                <i class="ri-edit-2-line align-bottom"></i> Edit
                                            </a>
                                            <?php endif; ?>
                                            <a href="quotations_list.php" class="btn btn-light btn-sm">
                                                <i class="ri-arrow-left-line align-bottom"></i> Back
                                            </a>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="card-body">
                                    <div class="row">
                                        <!-- Quotation Info -->
                                        <div class="col-sm-6">
                                            <div class="mb-4">
                                                <h5 class="text-muted mb-3">Quotation Information</h5>
                                                <table class="table table-sm table-borderless mb-0">
                                                    <tr>
                                                        <th width="140">Quotation No:</th>
                                                        <td><?php echo $quotation['quotation_no']; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Corporate:</th>
                                                        <td><?php echo htmlspecialchars($quotation['corporate_name']); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Scheme:</th>
                                                        <td><?php echo htmlspecialchars($quotation['scheme']); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Type:</th>
                                                        <td><?php echo $quotation['quotation_type']; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Valid Until:</th>
                                                        <td><?php echo date('d M Y', strtotime($quotation['valid_until'])); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Status:</th>
                                                        <td>
                                                            <span class="badge bg-<?php 
                                                                echo $quotation['status'] === 'PENDING' ? 'warning' : 
                                                                    ($quotation['status'] === 'ACCEPTED' ? 'success' : 'danger'); 
                                                            ?>">
                                                                <?php echo $quotation['status']; ?>
                                                            </span>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </div>
                                        
                                        <!-- Summary -->
                                        <div class="col-sm-6">
                                            <div class="mb-4">
                                                <h5 class="text-muted mb-3">Premium Summary</h5>
                                                <div class="bg-light p-3 rounded">
                                                    <table class="table table-sm table-borderless mb-0">
                                                        <tr>
                                                            <th>Total Members:</th>
                                                            <td class="text-end"><?php echo count($members); ?></td>
                                                        </tr>
                                                        <tr>
                                                            <th>Total Premium:</th>
                                                            <td class="text-end fw-bold h4 mb-0">
                                                                <?php echo number_format($quotation['total_premium'], 2); ?>
                                                            </td>
                                                        </tr>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Members & Benefits -->
                                    <div class="mt-4">
                                        <h5 class="text-muted mb-3">Members & Benefits</h5>
                                        <?php foreach($members as $member): ?>
                                            <div class="card border mb-3">
                                                <div class="card-header bg-light">
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div>
                                                            <span class="badge bg-primary me-2">
                                                                <?php echo ucfirst($member['relation']); ?>
                                                            </span>
                                                            Age: <?php echo $member['age']; ?>
                                                        </div>
                                                        <div class="text-end">
                                                            Premium: <?php echo number_format($member['premium'], 2); ?>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="card-body">
                                                    <div class="table-responsive">
                                                        <table class="table table-bordered mb-0">
                                                            <thead class="table-light">
                                                                <tr>
                                                                    <th>Benefit</th>
                                                                    <th class="text-end">Limit</th>
                                                                    <th class="text-end">Premium</th>
                                                                </tr>
                                                            </thead>
                                                            <tbody>
                                                                <?php foreach($member['benefits'] as $benefit): ?>
                                                                <tr>
                                                                    <td><?php echo htmlspecialchars($benefit['name']); ?></td>
                                                                    <td class="text-end">
                                                                        <?php echo number_format($benefit['limit'], 2); ?>
                                                                    </td>
                                                                    <td class="text-end">
                                                                        <?php echo number_format($benefit['premium'], 2); ?>
                                                                    </td>
                                                                </tr>
                                                                <?php endforeach; ?>
                                                            </tbody>
                                                        </table>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
</body>
</html>