<?php
session_start();
include 'layouts/dbconn.php';

// Set timezone to East Africa Time
date_default_timezone_set('Africa/Nairobi');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get role ID from URL
$role_id = $_GET['role_id'] ?? 0;

if (!$role_id) {
    header('Location: user_roles.php');
    exit();
}

// Fetch current user details
$current_user_id = $_SESSION['user_id'];
$sql_current_user = "SELECT u.*, ur.role_name FROM users u 
                     INNER JOIN user_roles ur ON u.role_id = ur.role_id 
                     WHERE u.user_id = ?";
$stmt_current = $conn->prepare($sql_current_user);
$stmt_current->bind_param("i", $current_user_id);
$stmt_current->execute();
$current_user = $stmt_current->get_result()->fetch_assoc();

// Fetch role details
$role_sql = "SELECT * FROM user_roles WHERE role_id = ?";
$role_stmt = $conn->prepare($role_sql);
$role_stmt->bind_param("i", $role_id);
$role_stmt->execute();
$role = $role_stmt->get_result()->fetch_assoc();

if (!$role) {
    header('Location: user_roles.php?error=Role not found');
    exit();
}

// Get search parameters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status_filter'] ?? '';
$branch_filter = $_GET['branch_filter'] ?? '';
$page = $_GET['page'] ?? 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Build query with filters
$where_conditions = array("u.role_id = ?");
$params = array($role_id);
$param_types = "i";

if (!empty($search)) {
    $where_conditions[] = "(u.full_name LIKE ? OR u.username LIKE ? OR u.email LIKE ? OR u.staff_number LIKE ?)";
    $search_param = "%$search%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
    $param_types .= "ssss";
}

if (!empty($status_filter)) {
    $where_conditions[] = "u.status = ?";
    $params[] = $status_filter;
    $param_types .= "s";
}

if (!empty($branch_filter)) {
    $where_conditions[] = "u.branch_id = ?";
    $params[] = $branch_filter;
    $param_types .= "i";
}

$where_clause = implode(" AND ", $where_conditions);

// Get total count for pagination
$count_query = "SELECT COUNT(*) as total FROM users u WHERE $where_clause";
$count_stmt = $conn->prepare($count_query);
if (!empty($params)) {
    $count_stmt->bind_param($param_types, ...$params);
}
$count_stmt->execute();
$total_users = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_users / $limit);

// Get users with pagination
$query = "SELECT u.*, b.branch_name, supervisor.full_name as supervisor_name
          FROM users u 
          LEFT JOIN branches b ON u.branch_id = b.branch_id 
          LEFT JOIN users supervisor ON u.supervisor_id = supervisor.user_id
          WHERE $where_clause
          ORDER BY u.created_at DESC 
          LIMIT ? OFFSET ?";

$stmt = $conn->prepare($query);
$params[] = $limit;
$params[] = $offset;
$param_types .= "ii";

if (!empty($params)) {
    $stmt->bind_param($param_types, ...$params);
}
$stmt->execute();
$users = $stmt->get_result();

// Get branches for filter dropdown
$branches_query = "SELECT branch_id, branch_name FROM branches WHERE is_active = 1 ORDER BY branch_name";
$branches_result = $conn->query($branches_query);

// Get role statistics
$stats_query = "SELECT 
                COUNT(*) as total_users,
                COUNT(CASE WHEN status = 'ACTIVE' THEN 1 END) as active_users,
                COUNT(CASE WHEN status = 'INACTIVE' THEN 1 END) as inactive_users,
                COUNT(CASE WHEN status = 'PENDING' THEN 1 END) as pending_users,
                COUNT(CASE WHEN status = 'SUSPENDED' THEN 1 END) as suspended_users,
                COUNT(CASE WHEN last_login IS NOT NULL THEN 1 END) as users_with_login,
                COUNT(CASE WHEN last_login >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as active_last_30_days
                FROM users WHERE role_id = ?";
$stats_stmt = $conn->prepare($stats_query);
$stats_stmt->bind_param("i", $role_id);
$stats_stmt->execute();
$stats = $stats_stmt->get_result()->fetch_assoc();

// Get performance data for doctors
$doctor_performance = [];
if (in_array($role_id, [3, 4])) { // Doctor or Senior Doctor
    $performance_query = "SELECT 
        u.user_id,
        u.full_name,
        COUNT(c.consultation_id) as total_consultations,
        COUNT(CASE WHEN c.status = 'COMPLETED' THEN 1 END) as completed_consultations,
        COUNT(CASE WHEN MONTH(c.started_at) = MONTH(CURRENT_DATE()) AND YEAR(c.started_at) = YEAR(CURRENT_DATE()) THEN 1 END) as month_consultations,
        COALESCE(SUM(CASE WHEN c.status = 'COMPLETED' THEN c.consultation_fee END), 0) as total_revenue,
        COALESCE(SUM(CASE WHEN c.status = 'COMPLETED' THEN c.doctor_commission END), 0) as total_commission,
        COUNT(DISTINCT c.patient_id) as unique_patients
        FROM users u
        LEFT JOIN doctors d ON u.user_id = d.user_id
        LEFT JOIN consultations c ON d.doctor_id = c.doctor_id
        WHERE u.role_id = ? AND u.status = 'ACTIVE'
        GROUP BY u.user_id, u.full_name
        ORDER BY total_revenue DESC";
    $performance_stmt = $conn->prepare($performance_query);
    $performance_stmt->bind_param("i", $role_id);
    $performance_stmt->execute();
    $doctor_performance = $performance_stmt->get_result();
}
?>

<?php include 'layouts/main.php'; ?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => $role['role_name'] . ' Users - Healthcare EMR System')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <style>
        /* Use the same styling from dashboard */
        body {
            background-color: #f8fafc;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        
        .page-content {
            padding: 1.5rem;
        }
        
        /* Page Header */
        .page-header {
            background: linear-gradient(135deg, #1e40af 0%, #3b82f6 100%);
            color: white;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 1rem;
        }
        
        .page-header-content {
            flex: 1;
        }
        
        .page-title {
            font-size: 1.75rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .page-subtitle {
            opacity: 0.9;
            margin: 0;
        }
        
        .role-badge {
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
            min-width: 120px;
        }
        
        .role-user-count {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.25rem;
        }
        
        .role-label {
            font-size: 0.875rem;
            opacity: 0.8;
        }
        
        /* Stats Cards */
        .stats-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: #64748b;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        /* Filter Card */
        .filter-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            margin-bottom: 2rem;
        }
        
        /* Table Card */
        .table-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .table-header {
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            background: #f8fafc;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .table-title {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
        }
        
        .data-table th {
            background: #f8fafc;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            font-size: 0.875rem;
            color: #64748b;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .data-table td {
            padding: 1rem;
            border-bottom: 1px solid #f1f5f9;
            font-size: 0.875rem;
        }
        
        .data-table tbody tr:hover {
            background: #f8fafc;
        }
        
        /* Status Badges */
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        .status-active { background: #dcfce7; color: #166534; }
        .status-inactive { background: #f1f5f9; color: #475569; }
        .status-suspended { background: #fee2e2; color: #991b1b; }
        .status-pending { background: #fef3c7; color: #92400e; }
        
        /* Action Buttons */
        .btn-action {
            padding: 0.375rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            margin: 0 0.25rem;
            transition: all 0.2s ease;
        }
        
        .btn-primary { background: #3b82f6; color: white; }
        .btn-primary:hover { background: #1d4ed8; color: white; }
        
        .btn-warning { background: #d97706; color: white; }
        .btn-warning:hover { background: #b45309; color: white; }
        
        /* Pagination */
        .pagination-wrapper {
            padding: 1.5rem;
            background: #f8fafc;
            border-top: 1px solid #e2e8f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 1rem;
        }
        
        .pagination-info {
            color: #64748b;
            font-size: 0.875rem;
        }
        
        .pagination {
            display: flex;
            gap: 0.5rem;
            margin: 0;
        }
        
        .page-link {
            padding: 0.5rem 0.75rem;
            background: white;
            border: 1px solid #e2e8f0;
            color: #374151;
            text-decoration: none;
            border-radius: 6px;
            font-size: 0.875rem;
        }
        
        .page-link:hover {
            background: #f8fafc;
            border-color: #3b82f6;
            color: #3b82f6;
        }
        
        .page-link.active {
            background: #3b82f6;
            border-color: #3b82f6;
            color: white;
        }
        
        /* Form Controls */
        .form-control, .form-select {
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem;
            font-size: 0.875rem;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .btn-primary {
            background: #3b82f6;
            color: white;
        }
        
        .btn-primary:hover {
            background: #1d4ed8;
        }
        
        .btn-outline-secondary {
            background: transparent;
            border: 1px solid #e2e8f0;
            color: #64748b;
        }
        
        .btn-outline-secondary:hover {
            background: #f8fafc;
            border-color: #64748b;
        }
        
        .btn-secondary {
            background: #64748b;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #475569;
            color: white;
        }
        
        /* Avatar */
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #3b82f6;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 0.875rem;
        }
        
        /* Performance Table */
        .performance-table {
            margin-top: 2rem;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .page-content {
                padding: 1rem;
            }
            
            .page-header {
                flex-direction: column;
                text-align: center;
            }
            
            .stats-row {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .table-header {
                flex-direction: column;
                gap: 1rem;
                align-items: stretch;
            }
            
            .data-table {
                font-size: 0.75rem;
            }
            
            .data-table th,
            .data-table td {
                padding: 0.5rem;
            }
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <!-- Page Header -->
                    <div class="page-header">
                        <div class="page-header-content">
                            <h1 class="page-title"><?php echo htmlspecialchars($role['role_name']); ?> Users</h1>
                            <p class="page-subtitle">
                                Manage users with the <?php echo htmlspecialchars($role['role_name']); ?> role
                                <?php if ($role['description']): ?>
                                    - <?php echo htmlspecialchars($role['description']); ?>
                                <?php endif; ?>
                            </p>
                        </div>
                        <div class="role-badge">
                            <div class="role-user-count"><?php echo $total_users; ?></div>
                            <div class="role-label">Total Users</div>
                        </div>
                    </div>

                    <!-- Stats Row -->
                    <div class="stats-row">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $stats['active_last_30_days']; ?></div>
                            <div class="stat-label">Active Last 30 Days</div>
                        </div>
                        <?php if ($stats['suspended_users'] > 0): ?>
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $stats['suspended_users']; ?></div>
                            <div class="stat-label">Suspended Users</div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Filter Card -->
                    <div class="filter-card">
                        <form method="GET" class="row g-3">
                            <input type="hidden" name="role_id" value="<?php echo $role_id; ?>">
                            <div class="col-md-4">
                                <label class="form-label">Search</label>
                                <input type="text" class="form-control" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search by name, username, email...">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Status</label>
                                <select class="form-select" name="status_filter">
                                    <option value="">All Status</option>
                                    <option value="ACTIVE" <?php echo $status_filter == 'ACTIVE' ? 'selected' : ''; ?>>Active</option>
                                    <option value="INACTIVE" <?php echo $status_filter == 'INACTIVE' ? 'selected' : ''; ?>>Inactive</option>
                                    <option value="PENDING" <?php echo $status_filter == 'PENDING' ? 'selected' : ''; ?>>Pending</option>
                                    <option value="SUSPENDED" <?php echo $status_filter == 'SUSPENDED' ? 'selected' : ''; ?>>Suspended</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Branch</label>
                                <select class="form-select" name="branch_filter">
                                    <option value="">All Branches</option>
                                    <?php while ($branch = $branches_result->fetch_assoc()): ?>
                                        <option value="<?php echo $branch['branch_id']; ?>" <?php echo $branch_filter == $branch['branch_id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($branch['branch_name']); ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">Filter</button>
                                    <a href="view_role_users.php?role_id=<?php echo $role_id; ?>" class="btn btn-outline-secondary">Clear</a>
                                </div>
                            </div>
                        </form>
                    </div>

                    <!-- Users Table -->
                    <div class="table-card">
                        <div class="table-header">
                            <h3 class="table-title"><?php echo htmlspecialchars($role['role_name']); ?> Users (<?php echo $total_users; ?> total)</h3>
                            <div class="d-flex gap-2">
                                <a href="add_user.php?role_id=<?php echo $role_id; ?>" class="btn btn-primary">
                                    <i class="ph-user-plus me-1"></i> Add <?php echo htmlspecialchars($role['role_name']); ?>
                                </a>
                                <a href="user_roles.php" class="btn btn-secondary">
                                    <i class="ph-arrow-left me-1"></i> Back to Roles
                                </a>
                            </div>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>User</th>
                                    <th>Contact</th>
                                    <th>Branch</th>
                                    <th>Supervisor</th>
                                    <th>Status</th>
                                    <th>Last Login</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($user = $users->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="user-avatar me-3">
                                                <?php echo strtoupper(substr($user['full_name'], 0, 2)); ?>
                                            </div>
                                            <div>
                                                <div style="font-weight: 600; color: #1e293b;">
                                                    <?php echo htmlspecialchars($user['full_name']); ?>
                                                </div>
                                                <div style="color: #64748b; font-size: 0.75rem;">
                                                    @<?php echo htmlspecialchars($user['username']); ?>
                                                    <?php if ($user['staff_number']): ?>
                                                        | <?php echo htmlspecialchars($user['staff_number']); ?>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div style="font-weight: 500; color: #1e293b;">
                                            <?php echo htmlspecialchars($user['email']); ?>
                                        </div>
                                        <?php if ($user['phone_number']): ?>
                                            <div style="color: #64748b; font-size: 0.75rem;">
                                                <?php echo htmlspecialchars($user['phone_number']); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($user['branch_name']): ?>
                                            <span style="color: #1e293b; font-weight: 500;">
                                                <?php echo htmlspecialchars($user['branch_name']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span style="color: #64748b;">Not assigned</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($user['supervisor_name']): ?>
                                            <span style="color: #1e293b; font-weight: 500;">
                                                <?php echo htmlspecialchars($user['supervisor_name']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span style="color: #64748b;">Not assigned</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        $status_class = match($user['status']) {
                                            'ACTIVE' => 'status-active',
                                            'INACTIVE' => 'status-inactive',
                                            'PENDING' => 'status-pending',
                                            'SUSPENDED' => 'status-suspended',
                                            default => 'status-inactive'
                                        };
                                        ?>
                                        <span class="status-badge <?php echo $status_class; ?>">
                                            <?php echo ucfirst(strtolower($user['status'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($user['last_login']): ?>
                                            <span style="color: #1e293b;">
                                                <?php echo date('M j, Y', strtotime($user['last_login'])); ?>
                                            </span>
                                            <div style="color: #64748b; font-size: 0.75rem;">
                                                <?php echo date('g:i A', strtotime($user['last_login'])); ?>
                                            </div>
                                        <?php else: ?>
                                            <span style="color: #64748b;">Never</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="d-flex gap-1">
                                            <a href="view_user.php?id=<?php echo $user['user_id']; ?>" class="btn-action btn-primary" title="View Details">
                                                <i class="ph-eye"></i>
                                            </a>
                                            <a href="edit_user.php?id=<?php echo $user['user_id']; ?>" class="btn-action btn-warning" title="Edit User">
                                                <i class="ph-pencil"></i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>

                        <!-- Pagination -->
                        <?php if ($total_pages > 1): ?>
                        <div class="pagination-wrapper">
                            <div class="pagination-info">
                                Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $limit, $total_users); ?> of <?php echo $total_users; ?> entries
                            </div>
                            
                            <div class="pagination">
                                <?php if ($page > 1): ?>
                                    <a href="?role_id=<?php echo $role_id; ?>&page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status_filter=<?php echo $status_filter; ?>&branch_filter=<?php echo $branch_filter; ?>" class="page-link">
                                        <i class="ph-caret-left"></i>
                                    </a>
                                <?php endif; ?>
                                
                                <?php 
                                $start = max(1, $page - 2);
                                $end = min($total_pages, $page + 2);
                                
                                for ($i = $start; $i <= $end; $i++): 
                                ?>
                                    <a href="?role_id=<?php echo $role_id; ?>&page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status_filter=<?php echo $status_filter; ?>&branch_filter=<?php echo $branch_filter; ?>" 
                                       class="page-link <?php echo $i == $page ? 'active' : ''; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                <?php endfor; ?>
                                
                                <?php if ($page < $total_pages): ?>
                                    <a href="?role_id=<?php echo $role_id; ?>&page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status_filter=<?php echo $status_filter; ?>&branch_filter=<?php echo $branch_filter; ?>" class="page-link">
                                        <i class="ph-caret-right"></i>
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Doctor Performance Table (if applicable) -->
                    <?php if (in_array($role_id, [3, 4]) && $doctor_performance->num_rows > 0): ?>
                    <div class="table-card performance-table">
                        <div class="table-header">
                            <h3 class="table-title">Doctor Performance Summary</h3>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Doctor</th>
                                    <th>Total Consultations</th>
                                    <th>This Month</th>
                                    <th>Completed</th>
                                    <th>Total Revenue</th>
                                    <th>Total Commission</th>
                                    <th>Unique Patients</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($performance = $doctor_performance->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <div style="font-weight: 600;">
                                            <a href="view_user.php?id=<?php echo $performance['user_id']; ?>" style="color: #3b82f6; text-decoration: none;">
                                                Dr. <?php echo htmlspecialchars($performance['full_name']); ?>
                                            </a>
                                        </div>
                                    </td>
                                    <td style="font-weight: 600;"><?php echo number_format($performance['total_consultations']); ?></td>
                                    <td style="font-weight: 600;"><?php echo number_format($performance['month_consultations']); ?></td>
                                    <td style="font-weight: 600;"><?php echo number_format($performance['completed_consultations']); ?></td>
                                    <td style="font-weight: 600; color: #059669;">KES <?php echo number_format($performance['total_revenue']); ?></td>
                                    <td style="font-weight: 600; color: #d97706;">KES <?php echo number_format($performance['total_commission']); ?></td>
                                    <td style="font-weight: 600;"><?php echo number_format($performance['unique_patients']); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Auto-submit form on filter change
            $('.form-select').on('change', function() {
                $(this).closest('form').submit();
            });
        });
    </script>

    <!-- App js -->
    <script src="assets/js/app.js"></script>
</body>
</html>

<?php $conn->close(); ?> 