<?php
session_start();
include 'layouts/dbconn.php';

// Set timezone to East Africa Time
date_default_timezone_set('Africa/Nairobi');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user ID from URL
$view_user_id = $_GET['id'] ?? 0;

if (!$view_user_id) {
    header('Location: list_users.php');
    exit();
}

// Fetch current user details
$current_user_id = $_SESSION['user_id'];
$sql_current_user = "SELECT u.*, ur.role_name FROM users u 
                     INNER JOIN user_roles ur ON u.role_id = ur.role_id 
                     WHERE u.user_id = ?";
$stmt_current = $conn->prepare($sql_current_user);
$stmt_current->bind_param("i", $current_user_id);
$stmt_current->execute();
$current_user = $stmt_current->get_result()->fetch_assoc();

// Fetch user details
$sql_user = "SELECT u.*, ur.role_name, b.branch_name, b.branch_code, 
             supervisor.full_name as supervisor_name
             FROM users u 
             INNER JOIN user_roles ur ON u.role_id = ur.role_id 
             LEFT JOIN branches b ON u.branch_id = b.branch_id 
             LEFT JOIN users supervisor ON u.supervisor_id = supervisor.user_id
             WHERE u.user_id = ?";
$stmt_user = $conn->prepare($sql_user);
$stmt_user->bind_param("i", $view_user_id);
$stmt_user->execute();
$user = $stmt_user->get_result()->fetch_assoc();

if (!$user) {
    header('Location: list_users.php?error=User not found');
    exit();
}

// Fetch doctor details if user is a doctor
$doctor_details = null;
if (in_array($user['role_id'], [3, 4])) { // Doctor or Senior Doctor
    $doctor_sql = "SELECT * FROM doctors WHERE user_id = ?";
    $doctor_stmt = $conn->prepare($doctor_sql);
    $doctor_stmt->bind_param("i", $view_user_id);
    $doctor_stmt->execute();
    $doctor_details = $doctor_stmt->get_result()->fetch_assoc();
}

// Fetch nurse details if user is a nurse
$nurse_details = null;
if ($user['role_id'] == 5) { // Nurse
    $nurse_sql = "SELECT * FROM nurses WHERE user_id = ?";
    $nurse_stmt = $conn->prepare($nurse_sql);
    $nurse_stmt->bind_param("i", $view_user_id);
    $nurse_stmt->execute();
    $nurse_details = $nurse_stmt->get_result()->fetch_assoc();
}

// Fetch recent activity
$activity_sql = "SELECT * FROM activity_log 
                 WHERE user_id = ? 
                 ORDER BY logged_at DESC 
                 LIMIT 10";
$activity_stmt = $conn->prepare($activity_sql);
$activity_stmt->bind_param("i", $view_user_id);
$activity_stmt->execute();
$recent_activity = $activity_stmt->get_result();

// Get performance stats based on role
$performance_stats = [];

if ($doctor_details) {
    // Doctor performance stats
    $doctor_stats_sql = "SELECT 
        COUNT(c.consultation_id) as total_consultations,
        COUNT(CASE WHEN c.status = 'COMPLETED' THEN 1 END) as completed_consultations,
        COUNT(CASE WHEN DATE(c.started_at) = CURDATE() THEN 1 END) as today_consultations,
        COUNT(CASE WHEN MONTH(c.started_at) = MONTH(CURRENT_DATE()) AND YEAR(c.started_at) = YEAR(CURRENT_DATE()) THEN 1 END) as month_consultations,
        COALESCE(SUM(CASE WHEN c.status = 'COMPLETED' THEN c.consultation_fee END), 0) as total_revenue,
        COALESCE(SUM(CASE WHEN c.status = 'COMPLETED' THEN c.doctor_commission END), 0) as total_commission,
        COUNT(DISTINCT c.patient_id) as unique_patients,
        COALESCE(AVG(CASE WHEN c.status = 'COMPLETED' THEN c.consultation_fee END), 0) as avg_consultation_fee
        FROM consultations c 
        WHERE c.doctor_id = ?";
    $doctor_stats_stmt = $conn->prepare($doctor_stats_sql);
    $doctor_stats_stmt->bind_param("i", $doctor_details['doctor_id']);
    $doctor_stats_stmt->execute();
    $performance_stats = $doctor_stats_stmt->get_result()->fetch_assoc();
}

// Get recent patients for doctors
$recent_patients = [];
if ($doctor_details) {
    $patients_sql = "SELECT DISTINCT p.patient_number, p.first_name, p.last_name, 
                     c.started_at, c.status, c.consultation_fee
                     FROM consultations c
                     JOIN patients p ON c.patient_id = p.patient_id
                     WHERE c.doctor_id = ?
                     ORDER BY c.started_at DESC
                     LIMIT 5";
    $patients_stmt = $conn->prepare($patients_sql);
    $patients_stmt->bind_param("i", $doctor_details['doctor_id']);
    $patients_stmt->execute();
    $recent_patients = $patients_stmt->get_result();
}
?>

<?php include 'layouts/main.php'; ?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'View User - Healthcare EMR System')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <style>
        /* Use the same styling from dashboard */
        body {
            background-color: #f8fafc;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        
        .page-content {
            padding: 1.5rem;
        }
        
        /* Page Header */
        .page-header {
            background: linear-gradient(135deg, #1e40af 0%, #3b82f6 100%);
            color: white;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .page-title {
            font-size: 1.75rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .page-subtitle {
            opacity: 0.9;
            margin: 0;
        }
        
        /* User Profile Card */
        .profile-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .profile-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            padding: 2rem;
            text-align: center;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .user-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: #3b82f6;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 2rem;
            margin: 0 auto 1rem;
            border: 4px solid white;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .user-name {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        
        .user-role {
            display: inline-block;
            background: #3b82f6;
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.875rem;
            font-weight: 600;
            margin-bottom: 1rem;
        }
        
        .user-basic-info {
            display: flex;
            justify-content: center;
            gap: 2rem;
            flex-wrap: wrap;
            color: #64748b;
            font-size: 0.875rem;
        }
        
        .user-basic-info .info-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        /* Info Cards */
        .info-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .info-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            border: 1px solid #e2e8f0;
            overflow: hidden;
        }
        
        .info-card-header {
            padding: 1.5rem 1.5rem 0;
        }
        
        .info-card-title {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0 0 1rem 0;
        }
        
        .info-card-body {
            padding: 0 1.5rem 1.5rem;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem 0;
            border-bottom: 1px solid #f1f5f9;
        }
        
        .info-row:last-child {
            border-bottom: none;
        }
        
        .info-label {
            color: #64748b;
            font-size: 0.875rem;
            font-weight: 500;
        }
        
        .info-value {
            color: #1e293b;
            font-weight: 600;
            text-align: right;
        }
        
        /* Status Badges */
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        .status-active { background: #dcfce7; color: #166534; }
        .status-inactive { background: #f1f5f9; color: #475569; }
        .status-suspended { background: #fee2e2; color: #991b1b; }
        .status-pending { background: #fef3c7; color: #92400e; }
        
        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .stat-card {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1.5rem;
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: #64748b;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        
        /* Table */
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
        }
        
        .data-table th {
            background: #f8fafc;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            font-size: 0.875rem;
            color: #64748b;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .data-table td {
            padding: 1rem;
            border-bottom: 1px solid #f1f5f9;
            font-size: 0.875rem;
        }
        
        .data-table tbody tr:hover {
            background: #f8fafc;
        }
        
        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn-primary {
            background: #3b82f6;
            color: white;
        }
        
        .btn-primary:hover {
            background: #1d4ed8;
            color: white;
        }
        
        .btn-warning {
            background: #d97706;
            color: white;
        }
        
        .btn-warning:hover {
            background: #b45309;
            color: white;
        }
        
        .btn-danger {
            background: #dc2626;
            color: white;
        }
        
        .btn-danger:hover {
            background: #b91c1c;
            color: white;
        }
        
        .btn-secondary {
            background: #64748b;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #475569;
            color: white;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .page-content {
                padding: 1rem;
            }
            
            .info-cards {
                grid-template-columns: 1fr;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .user-basic-info {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }
        
        /* Footer spacing fix */
        .main-content {
            padding-bottom: 2rem;
        }
        
        .page-content {
            margin-bottom: 3rem;
        }
    </style>
</head>

<body class="view-user-page user-management-page">
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <!-- Page Header -->
                    <div class="page-header">
                        <h1 class="page-title">User Profile</h1>
                        <p class="page-subtitle">View detailed user information and activity</p>
                    </div>

                    <!-- User Profile Card -->
                    <div class="profile-card">
                        <div class="profile-header">
                            <div class="user-avatar">
                                <?php echo strtoupper(substr($user['full_name'], 0, 2)); ?>
                            </div>
                            <div class="user-name"><?php echo htmlspecialchars($user['full_name']); ?></div>
                            <div class="user-role"><?php echo htmlspecialchars($user['role_name']); ?></div>
                            <div class="user-basic-info">
                                <div class="info-item">
                                    <i class="ph-envelope"></i>
                                    <?php echo htmlspecialchars($user['email']); ?>
                                </div>
                                <div class="info-item">
                                    <i class="ph-phone"></i>
                                    <?php echo htmlspecialchars($user['phone_number']); ?>
                                </div>
                                <div class="info-item">
                                    <i class="ph-calendar"></i>
                                    Joined <?php echo date('M Y', strtotime($user['created_at'])); ?>
                                </div>
                                <div class="info-item">
                                    <?php 
                                    $status_class = match($user['status']) {
                                        'ACTIVE' => 'status-active',
                                        'INACTIVE' => 'status-inactive',
                                        'PENDING' => 'status-pending',
                                        'SUSPENDED' => 'status-suspended',
                                        default => 'status-inactive'
                                    };
                                    ?>
                                    <span class="status-badge <?php echo $status_class; ?>">
                                        <?php echo ucfirst(strtolower($user['status'])); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Performance Stats for Doctors -->
                    <?php if ($doctor_details && !empty($performance_stats)): ?>
                    <div class="info-card">
                        <div class="info-card-header">
                            <h3 class="info-card-title">Performance Statistics</h3>
                        </div>
                        <div class="info-card-body">
                            <div class="stats-grid">
                                <div class="stat-card">
                                    <div class="stat-value"><?php echo $performance_stats['total_consultations']; ?></div>
                                    <div class="stat-label">Total Consultations</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-value"><?php echo $performance_stats['month_consultations']; ?></div>
                                    <div class="stat-label">This Month</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-value"><?php echo $performance_stats['unique_patients']; ?></div>
                                    <div class="stat-label">Unique Patients</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-value">KES <?php echo number_format($performance_stats['total_revenue']); ?></div>
                                    <div class="stat-label">Total Revenue</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-value">KES <?php echo number_format($performance_stats['total_commission']); ?></div>
                                    <div class="stat-label">Total Commission</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-value">KES <?php echo number_format($performance_stats['avg_consultation_fee']); ?></div>
                                    <div class="stat-label">Avg Fee</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Information Cards -->
                    <div class="info-cards">
                        <!-- Personal Information -->
                        <div class="info-card">
                            <div class="info-card-header">
                                <h3 class="info-card-title">Personal Information</h3>
                            </div>
                            <div class="info-card-body">
                                <div class="info-row">
                                    <span class="info-label">Username</span>
                                    <span class="info-value">@<?php echo htmlspecialchars($user['username']); ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Date of Birth</span>
                                    <span class="info-value">
                                        <?php echo $user['date_of_birth'] ? date('M j, Y', strtotime($user['date_of_birth'])) : 'Not provided'; ?>
                                    </span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Gender</span>
                                    <span class="info-value">
                                        <?php echo $user['gender'] ? ucfirst(strtolower($user['gender'])) : 'Not specified'; ?>
                                    </span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">National ID</span>
                                    <span class="info-value">
                                        <?php echo $user['national_id'] ?: 'Not provided'; ?>
                                    </span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Address</span>
                                    <span class="info-value">
                                        <?php echo $user['address'] ?: 'Not provided'; ?>
                                    </span>
                                </div>
                            </div>
                        </div>

                        <!-- Work Information -->
                        <div class="info-card">
                            <div class="info-card-header">
                                <h3 class="info-card-title">Work Information</h3>
                            </div>
                            <div class="info-card-body">
                                <div class="info-row">
                                    <span class="info-label">Staff Number</span>
                                    <span class="info-value"><?php echo $user['staff_number'] ?: 'Not assigned'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Branch</span>
                                    <span class="info-value"><?php echo $user['branch_name'] ?: 'Not assigned'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Supervisor</span>
                                    <span class="info-value"><?php echo $user['supervisor_name'] ?: 'Not assigned'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Commission Rate</span>
                                    <span class="info-value"><?php echo $user['commission_rate']; ?>%</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Hire Date</span>
                                    <span class="info-value">
                                        <?php echo $user['hire_date'] ? date('M j, Y', strtotime($user['hire_date'])) : 'Not specified'; ?>
                                    </span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Last Login</span>
                                    <span class="info-value">
                                        <?php echo $user['last_login'] ? date('M j, Y g:i A', strtotime($user['last_login'])) : 'Never'; ?>
                                    </span>
                                </div>
                            </div>
                        </div>

                        <!-- License Information -->
                        <?php if ($user['license_number'] || $doctor_details || $nurse_details): ?>
                        <div class="info-card">
                            <div class="info-card-header">
                                <h3 class="info-card-title">Professional Information</h3>
                            </div>
                            <div class="info-card-body">
                                <div class="info-row">
                                    <span class="info-label">License Number</span>
                                    <span class="info-value"><?php echo $user['license_number'] ?: 'Not provided'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">License Expiry</span>
                                    <span class="info-value">
                                        <?php echo $user['license_expiry'] ? date('M j, Y', strtotime($user['license_expiry'])) : 'Not specified'; ?>
                                    </span>
                                </div>
                                <?php if ($doctor_details): ?>
                                <div class="info-row">
                                    <span class="info-label">Specialization</span>
                                    <span class="info-value"><?php echo $doctor_details['specialization'] ?: 'General Practice'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Consultation Fee</span>
                                    <span class="info-value">KES <?php echo number_format($doctor_details['consultation_fee']); ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Experience</span>
                                    <span class="info-value"><?php echo $doctor_details['experience_years']; ?> years</span>
                                </div>
                                <?php endif; ?>
                                <?php if ($nurse_details): ?>
                                <div class="info-row">
                                    <span class="info-label">Qualification</span>
                                    <span class="info-value"><?php echo $nurse_details['qualification'] ?: 'Not specified'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Shift Type</span>
                                    <span class="info-value"><?php echo ucfirst(strtolower($nurse_details['shift_type'])); ?></span>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Emergency Contact -->
                        <?php if ($user['emergency_contact'] || $user['emergency_phone']): ?>
                        <div class="info-card">
                            <div class="info-card-header">
                                <h3 class="info-card-title">Emergency Contact</h3>
                            </div>
                            <div class="info-card-body">
                                <div class="info-row">
                                    <span class="info-label">Contact Name</span>
                                    <span class="info-value"><?php echo $user['emergency_contact'] ?: 'Not provided'; ?></span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Contact Phone</span>
                                    <span class="info-value"><?php echo $user['emergency_phone'] ?: 'Not provided'; ?></span>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Recent Patients (for doctors) -->
                    <?php if ($doctor_details && $recent_patients->num_rows > 0): ?>
                    <div class="info-card">
                        <div class="info-card-header">
                            <h3 class="info-card-title">Recent Patients</h3>
                        </div>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Patient</th>
                                    <th>Consultation Date</th>
                                    <th>Status</th>
                                    <th>Fee</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($patient = $recent_patients->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <div style="font-weight: 600;"><?php echo htmlspecialchars($patient['first_name'] . ' ' . $patient['last_name']); ?></div>
                                        <div style="color: #64748b; font-size: 0.75rem;"><?php echo $patient['patient_number']; ?></div>
                                    </td>
                                    <td><?php echo date('M j, Y g:i A', strtotime($patient['started_at'])); ?></td>
                                    <td>
                                        <?php 
                                        $status_class = match($patient['status']) {
                                            'COMPLETED' => 'status-active',
                                            'IN_PROGRESS' => 'status-pending',
                                            'CANCELLED' => 'status-inactive',
                                            default => 'status-inactive'
                                        };
                                        ?>
                                        <span class="status-badge <?php echo $status_class; ?>">
                                            <?php echo ucfirst(strtolower(str_replace('_', ' ', $patient['status']))); ?>
                                        </span>
                                    </td>
                                    <td style="font-weight: 600;">KES <?php echo number_format($patient['consultation_fee']); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>

                    <!-- Recent Activity -->
                    <?php if ($recent_activity->num_rows > 0): ?>
                    <div class="info-card">
                        <div class="info-card-header">
                            <h3 class="info-card-title">Recent Activity</h3>
                        </div>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Action</th>
                                    <th>Module</th>
                                    <th>Date & Time</th>
                                    <th>IP Address</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($activity = $recent_activity->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <span style="font-weight: 600; color: #1e293b;">
                                            <?php echo htmlspecialchars($activity['action']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($activity['module']); ?></td>
                                    <td><?php echo date('M j, Y g:i A', strtotime($activity['logged_at'])); ?></td>
                                    <td style="color: #64748b;"><?php echo htmlspecialchars($activity['ip_address']); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>

                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <a href="edit_user.php?id=<?php echo $user['user_id']; ?>" class="btn btn-warning">
                            <i class="ph-pencil me-1"></i> Edit User
                        </a>
                        
                        <?php if ($user['status'] == 'ACTIVE'): ?>
                            <a href="suspend_user.php?id=<?php echo $user['user_id']; ?>" class="btn btn-danger" 
                               onclick="return confirm('Are you sure you want to suspend this user?')">
                                <i class="ph-prohibit me-1"></i> Suspend User
                            </a>
                        <?php else: ?>
                            <a href="activate_user.php?id=<?php echo $user['user_id']; ?>" class="btn btn-primary">
                                <i class="ph-check me-1"></i> Activate User
                            </a>
                        <?php endif; ?>
                        
                        <a href="reset_password.php?id=<?php echo $user['user_id']; ?>" class="btn btn-warning"
                           onclick="return confirm('Are you sure you want to reset this user\'s password?')">
                            <i class="ph-lock-key me-1"></i> Reset Password
                        </a>
                        
                        <a href="list_users.php" class="btn btn-secondary">
                            <i class="ph-arrow-left me-1"></i> Back to Users
                        </a>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Add any JavaScript functionality here
            console.log('View User page loaded');
        });
    </script>

    <!-- App js -->
    <script src="assets/js/app.js"></script>
</body>
</html>

<?php $conn->close(); ?>